/*
HMGLText.m

Author: Makoto Kinoshita

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "HMGLText.h"

@interface HMGLText (private)
// Drawing
- (void)_generateTexture;
- (void)_deleteTexture;
@end

@implementation HMGLText

//--------------------------------------------------------------//
#pragma mark -- Initialize --
//--------------------------------------------------------------//

- (id)initWithString:(NSString*)string withAttributes:(NSDictionary*)attrs
{
	return [self initWithAttributedString:
            [[[NSAttributedString alloc] initWithString:string attributes:attrs] autorelease]];
}

- (id)initWithAttributedString:(NSAttributedString*)attrStr
{
	self = [super init];
    if (!self) {
        return nil;
    }
    
    // Initialize instance variables
	_text = [attrStr retain];
	_textureId = 0;
    
    return self;
}

- (void) dealloc
{
	[self _deleteTexture];
	[_text release];
    
	[super dealloc];
}

//--------------------------------------------------------------//
#pragma mark -- Text --
//--------------------------------------------------------------//

- (void)setString:(NSString*)string withAttributes:(NSDictionary*)attrs
{
	[self setAttributedString:[[[NSAttributedString alloc] 
            initWithString:string attributes:attrs] autorelease]];
}

- (void)setAttributedString:(NSAttributedString*)attrStr
{
    // Delete texture
	[self _deleteTexture];
    
    // Set text
    [_text release];
    _text = [attrStr retain];
}

- (NSAttributedString*)text
{
    return _text;
}

//--------------------------------------------------------------//
#pragma mark -- Drawing --
//--------------------------------------------------------------//

- (void)_generateTexture
{
    // Delete current texture
    [self _deleteTexture];
    
    // Get text size
    NSSize  size;
    size = [_text size];
    
    // Create image
    NSImage*            image;
    NSBitmapImageRep*   bitmapRep;
    image = [[NSImage alloc] initWithSize:size];
    [image lockFocus];
    [_text drawAtPoint:NSZeroPoint];
    bitmapRep = [[NSBitmapImageRep alloc] 
            initWithFocusedViewRect:NSMakeRect(0, 0, size.width, size.height)];
    [image unlockFocus];
    
    // Create texture
    glEnable(GL_TEXTURE_RECTANGLE_EXT);
    glGenTextures(1, &_textureId);
    glBindTexture(GL_TEXTURE_RECTANGLE_EXT, _textureId);
    glTexImage2D(
            GL_TEXTURE_RECTANGLE_EXT, 0, GL_RGBA, 
            size.width, size.height, 0, GL_RGBA, GL_UNSIGNED_BYTE, [bitmapRep bitmapData]);
    glDisable(GL_TEXTURE_RECTANGLE_EXT);
    [bitmapRep release];
    [image release];
}

- (void)_deleteTexture
{
    if (!_textureId) {
        return;
    }
    
    // Delete texture
    glDeleteTextures(1, &_textureId);
    _textureId = 0;
}

- (void)drawAtPoint:(NSPoint)point
{
    // Generate texture
    if (!_textureId) {
        [self _generateTexture];
    }
    
    // Draw with bounds
    NSSize  size;
    size = [_text size];
    [self drawWithBounds:NSMakeRect(point.x, point.y, size.width, size.height)];
}

- (void)drawWithBounds:(NSRect)bounds
{
    // Generate texture
    if (!_textureId) {
        [self _generateTexture];
    }
    
    // Get size
    NSSize  size;
    size = [_text size];
    
    // Draw texture
    glEnable(GL_TEXTURE_RECTANGLE_EXT);
    glBindTexture(GL_TEXTURE_RECTANGLE_EXT, _textureId);
    glBegin(GL_QUADS);
        // Upper left
        glTexCoord2f(0.0f, 0.0f);
        glVertex2f(bounds.origin.x, bounds.origin.y);
        
        // Lower left
        glTexCoord2f(0.0f, size.height);
        glVertex2f(bounds.origin.x, bounds.origin.y + bounds.size.height);
        
        // Upper right
        glTexCoord2f(size.width, size.height);
        glVertex2f(bounds.origin.x + bounds.size.width, bounds.origin.y + bounds.size.height);
        
        // Lower right
        glTexCoord2f(size.width, 0.0f);
        glVertex2f(bounds.origin.x + bounds.size.width, bounds.origin.y);
    glEnd();
    glDisable(GL_TEXTURE_RECTANGLE_EXT);
}

@end
