/*
RSSDAppController.m

Author: Makoto Kinoshita

Copyright 2004-2007 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "RSSPersistentStack.h"
#import "RSSFeedParser.h"

#import "RSSSyndication.h"
#import "RSSDAppController.h"

@implementation RSSDAppController

//--------------------------------------------------------------//
#pragma mark -- Initialize --
//--------------------------------------------------------------//

- (id)init
{
    self = [super init];
    if (!self) {
        return nil;
    }
    
    // Initialize instance variables
    _feedURLs = [[NSMutableSet set] retain];
    _numberOfFeeds = 0;
    _tmpXMLFilePathDict = [[NSMutableDictionary dictionary] retain];
    _initialRequestURLDict = [[NSMutableDictionary dictionary] retain];
    _downloadersCount = 0;
    _willNotify = NO;
    _refreshedTitles = [[NSMutableArray array] retain];
    
    return self;
}

- (void)dealloc
{
    [_feedURLs release], _feedURLs = nil;
    [_tmpXMLFilePathDict release], _tmpXMLFilePathDict = nil;
    [_initialRequestURLDict release], _initialRequestURLDict = nil;
    [_refreshedTitles release], _refreshedTitles = nil;
    
    [super dealloc];
}

//--------------------------------------------------------------//
#pragma mark -- Refresh feed --
//--------------------------------------------------------------//

- (void)_refreshNextFeeds
{
    // Get max connection number
    int maxNumber;
    maxNumber = [[NSUserDefaults standardUserDefaults] integerForKey:RSSDMaxConnectionNumber];
    if (maxNumber <= 0 || maxNumber > 16) {
        maxNumber = 4;
    }
    
    // Refresh feeds
    NSMutableArray* sentFeedURLs;
    NSEnumerator*   enumerator;
    NSString*       feedURL;
    sentFeedURLs = [NSMutableArray array];
    enumerator = [_feedURLs objectEnumerator];
    while ((feedURL = [enumerator nextObject]) && _downloadersCount < maxNumber) {
        // Download specified URL
        NSURLRequest*   request;
        NSURLDownload*  download;
        request = [NSURLRequest requestWithURL:[NSURL URLWithString:feedURL]];
        download = [[NSURLDownload alloc] initWithRequest:request delegate:self];
        
        [_initialRequestURLDict setObject:feedURL 
                forKey:[NSNumber numberWithUnsignedInt:(unsigned int)download]];
        
        _downloadersCount++;
        [sentFeedURLs addObject:feedURL];
    }
    
    // Remove sent feed URLs
    enumerator = [sentFeedURLs objectEnumerator];
    while (feedURL = [enumerator nextObject]) {
        [_feedURLs removeObject:feedURL];
    }
}

- (void)_refreshFeeds:(NSArray*)feedURLs
{
    // Remove old feeds
    [_feedURLs removeAllObjects];
    [_refreshedTitles removeAllObjects];
    
    // Copy feeds
    [_feedURLs addObjectsFromArray:feedURLs];
    _numberOfFeeds = [_feedURLs count];
    
    // Notify will start refresh
    NSDictionary*   userInfo;
    userInfo = [NSDictionary dictionaryWithObjectsAndKeys:
            [NSNumber numberWithInt:_numberOfFeeds], @"count", 
            nil];
    [[NSDistributedNotificationCenter defaultCenter] 
            postNotificationName:RSSDFeedWillStartRefresh object:nil userInfo:userInfo];
    
    // Refresh next feeds
    [self _refreshNextFeeds];
}

- (void)refreshAllFeeds
{
    NSMutableArray* feedURLs;
    feedURLs = [NSMutableArray array];
    
    // Get managed object context
    NSManagedObjectContext* context;
    context = [[RSSPersistentStack sharedInstance] managedObjectContext];
    
    // Get feeds
    NSFetchRequest* request;
    NSArray*        feeds;
    request = [[NSFetchRequest alloc] init];
    [request autorelease];
    [request setEntity:
            [NSEntityDescription entityForName:@"RSSFeed" inManagedObjectContext:context]];
    
    feeds = [context executeFetchRequest:request error:NULL];
    
    // Copy all feed URLs
    NSEnumerator*   enumerator;
    id              feed;
    enumerator = [feeds objectEnumerator];
    while (feed = [enumerator nextObject]) {
        // Get URL
        NSString*   urlString;
        urlString = [feed valueForKey:@"feedURL"];
        if (!urlString) {
            continue;
        }
        
        // Add URL
        [feedURLs addObject:urlString];
    }
    
    // Refresh feeds
    [self _refreshFeeds:feedURLs];
}

//--------------------------------------------------------------//
#pragma mark -- RSS syndication management --
//--------------------------------------------------------------//

- (void)terminate
{
    // Save managed object context
    [[RSSPersistentStack sharedInstance] save];
    
    // Stop timer
    if (_notifyTimer) {
        [_notifyTimer invalidate];
    }
    
    // Terminate itself
    [[NSApplication sharedApplication] terminate:self];
}

- (void)notifyTimerFired:(NSTimer*)timer
{
    // Notify progress
    NSDictionary*   userInfo;
    userInfo = [NSDictionary dictionaryWithObjectsAndKeys:
            [NSNumber numberWithInt:_numberOfFeeds - [_feedURLs count]], @"progress", 
            _refreshedTitles, @"refreshedTitles", 
            nil];
    [[NSDistributedNotificationCenter defaultCenter] 
            postNotificationName:RSSDFeedProgressRefresh object:nil userInfo:userInfo];
    
    // Clear progress
    [_refreshedTitles removeAllObjects];
    _notifyTimer = nil;
    _willNotify = NO;
}

//--------------------------------------------------------------//
#pragma mark -- NSApplication delegate --
//--------------------------------------------------------------//

- (void)applicationDidFinishLaunching:(NSNotification*)notification
{
    // Refresh feed
    if ([RSSDFeedURLs count] > 0) {
        // Refresh feeds
        [self _refreshFeeds:RSSDFeedURLs];
    }
    // Refresh all feeds
    else {
        [self refreshAllFeeds];
    }
}

//--------------------------------------------------------------//
#pragma mark -- NSURLDownload delegate --
//--------------------------------------------------------------//

- (void)download:(NSURLDownload*)download decideDestinationWithSuggestedFilename:(NSString*)filename
{
    // Create UUID string
    NSString*   uuidString;
    uuidString = HMCreateUUID();
    
    // Create tmp path
    NSString*   tmpPathDir;
    NSString*   tmpPath;
    tmpPathDir = [NSTemporaryDirectory() 
            stringByAppendingPathComponent:[NSString stringWithFormat:@"Shiira"]];
    if (![[NSFileManager defaultManager] fileExistsAtPath:tmpPathDir]) {
        // Create directory
        [[NSFileManager defaultManager] createDirectoryAtPath:tmpPathDir attributes:nil];
    }
    tmpPath = [tmpPathDir stringByAppendingPathComponent:uuidString];
    
    [_tmpXMLFilePathDict setObject:tmpPath 
            forKey:[NSNumber numberWithUnsignedInt:(unsigned int)download]];
    
    // Set detination file path
    [download setDestination:tmpPath allowOverwrite:YES];
}

- (void)_finishDownload:(NSURLDownload*)download
{
    // Decrement number of downloads
    _downloadersCount--;
    
    // Remove tmp file
    NSString*   tmpPath;
    tmpPath = [_tmpXMLFilePathDict objectForKey:
            [NSNumber numberWithUnsignedInt:(unsigned int)download]];
    if (tmpPath) {
        if ([[NSFileManager defaultManager] fileExistsAtPath:tmpPath]) {
            [[NSFileManager defaultManager] removeFileAtPath:tmpPath handler:NULL];
        }
    }
    
    // Remove download pointer from dictionary
//    [_tmpXMLFilePathDict removeObjectForKey:
//            [NSNumber numberWithUnsignedInt:(unsigned int)download]];
//    [_initialRequestURLDict removeObjectForKey:
//            [NSNumber numberWithUnsignedInt:(unsigned int)download]];
    
    // Notify refresh finish
    if (!_willNotify) {
        // Start timer
        _willNotify = YES;
        _notifyTimer = [NSTimer scheduledTimerWithTimeInterval:2.0f 
                target:self 
                selector:@selector(notifyTimerFired:) 
                userInfo:nil 
                repeats:NO];
    }
    
    // Refresh next feeds
    [self _refreshNextFeeds];
    
    // Check count
    if (_downloadersCount == 0) {
        // Notify did end refresh
        [[NSDistributedNotificationCenter defaultCenter] 
                postNotificationName:RSSDFeedDidEndRefresh object:nil];
        
        // Terminate itself
        [self terminate];
    }
}

- (void)downloadDidFinish:(NSURLDownload*)download
{
    // Get managed object context
    NSManagedObjectContext* context;
    context = [[RSSPersistentStack sharedInstance] managedObjectContext];
    
    // Get URL
    NSString*   urlString;
    urlString = [_initialRequestURLDict objectForKey:
            [NSNumber numberWithUnsignedInt:(unsigned int)download]];
    if (!urlString) {
        return;
    }
    
    // Get tmp path
    NSString*   tmpPath;
    tmpPath = [_tmpXMLFilePathDict objectForKey:
            [NSNumber numberWithUnsignedInt:(unsigned int)download]];
    if (!tmpPath) {
        return;
    }
    
    // Create XML document
    NSXMLDocument*  document;
    document = [[NSXMLDocument alloc] initWithContentsOfURL:[NSURL fileURLWithPath:tmpPath] 
            options:0 error:NULL];
    
    // Parse it
    RSSFeedParser*  parser;
    id              feed;
    parser = [[RSSFeedParser alloc] initWithXMLNode:[document rootElement] URLString:urlString];
    feed = [parser parseWithManagedObjectContext:context];
    
    // Keep feed title
    NSString*   title;
    title = [parser title];
    if (!title) {
        title = NSLocalizedString(@"Untitled", nil);
    }
    [_refreshedTitles addObject:title];
    [parser release];
    
    // Finish download
    [self _finishDownload:download];
}

- (void)download:(NSURLDownload*)download didFailWithError:(NSError*)error
{
    [self _finishDownload:download];
}

@end
