/**********************************************************************
Copyright (c) 2004 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 

Contributors:
2005 Andy Jefferson - added sin, cos, tan, asin, acos, atan methods
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import java.math.BigDecimal;
import java.math.BigInteger;

/**
 * Representation of Math functions in query languages.
 */
public class MathExpression extends ScalarExpression
{
    /**
     * @param qs The query statement
     */
    public MathExpression(QueryExpression qs)
    {
        super(qs);
    }

    /**
     * Returns the absolute value of the argument.
     * @param expr the expression
     * @return the result in a ScalarExpression instance
     */
    public ScalarExpression absMethod(ScalarExpression expr)
    {
        if (expr == null)
        {
            return new NullLiteral(qs);
        }
        if (expr instanceof Literal)
        {
            if (expr instanceof ByteLiteral)
            {
                int originalValue = ((BigInteger) ((ByteLiteral) expr).getValue()).intValue();
                BigInteger absValue = new BigInteger(String.valueOf(Math.abs(originalValue)));
                return new ByteLiteral(qs, expr.mapping, absValue);
            }
            else if (expr instanceof IntegerLiteral)
            {
                int originalValue = ((Number) ((IntegerLiteral) expr).getValue()).intValue();
                Integer absValue = new Integer(Math.abs(originalValue));
                return new IntegerLiteral(qs, expr.mapping, absValue);
            }
            else if (expr instanceof FloatingPointLiteral)
            {
                double originalValue = ((BigDecimal) ((FloatingPointLiteral) expr).getValue()).doubleValue();
                Double absValue = new Double(Math.abs(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, absValue);
            }
            throw new IllegalOperationException(this, "absMethod", expr);
        }
        else
        {
            return qs.getStoreManager().getDatastoreAdapter().getNumericExpressionForMethod("abs", expr);
        }
    }

    /**
     * Returns the square root of the argument.
     * @param expr the expression
     * @return the result in a ScalarExpression instance
     */
    public ScalarExpression sqrtMethod(ScalarExpression expr)
    {
        if (expr == null)
        {
            return new NullLiteral(qs);
        }
        if (expr instanceof Literal)
        {
            if (expr instanceof ByteLiteral)
            {
                int originalValue = ((BigInteger) ((ByteLiteral) expr).getValue()).intValue();
                Double newValue = new Double(Math.sqrt(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            else if (expr instanceof IntegerLiteral)
            {
                int originalValue = ((Number) ((IntegerLiteral) expr).getValue()).intValue();
                Double newValue = new Double(Math.sqrt(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            else if (expr instanceof FloatingPointLiteral)
            {
                double originalValue = ((BigDecimal) ((FloatingPointLiteral) expr).getValue()).doubleValue();
                Double newValue = new Double(Math.sqrt(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            throw new IllegalOperationException(this, "sqrtMethod", expr);
        }
        else
        {
            return qs.getStoreManager().getDatastoreAdapter().getNumericExpressionForMethod("sqrt", expr);
        }
    }

    /**
     * Returns the cosine of the argument.
     * @param expr the expression
     * @return the result in a ScalarExpression instance
     */
    public ScalarExpression cosMethod(ScalarExpression expr)
    {
        if (expr == null)
        {
            return new NullLiteral(qs);
        }
        if (expr instanceof Literal)
        {
            if (expr instanceof ByteLiteral)
            {
                int originalValue = ((BigInteger) ((ByteLiteral) expr).getValue()).intValue();
                Double newValue = new Double(Math.cos(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            else if (expr instanceof IntegerLiteral)
            {
                int originalValue = ((Number) ((IntegerLiteral) expr).getValue()).intValue();
                Double newValue = new Double(Math.cos(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            else if (expr instanceof FloatingPointLiteral)
            {
                double originalValue = ((BigDecimal) ((FloatingPointLiteral) expr).getValue()).doubleValue();
                Double newValue = new Double(Math.cos(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            throw new IllegalOperationException(this, "cosMethod", expr);
        }
        else
        {
            return qs.getStoreManager().getDatastoreAdapter().getNumericExpressionForMethod("cos", expr);
        }
    }

    /**
     * Returns the sine of the argument.
     * @param expr the expression
     * @return the result in a ScalarExpression instance
     */
    public ScalarExpression sinMethod(ScalarExpression expr)
    {
        if (expr == null)
        {
            return new NullLiteral(qs);
        }
        if (expr instanceof Literal)
        {
            if (expr instanceof ByteLiteral)
            {
                int originalValue = ((BigInteger) ((ByteLiteral) expr).getValue()).intValue();
                Double newValue = new Double(Math.sin(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            else if (expr instanceof IntegerLiteral)
            {
                int originalValue = ((Number) ((IntegerLiteral) expr).getValue()).intValue();
                Double newValue = new Double(Math.sin(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            else if (expr instanceof FloatingPointLiteral)
            {
                double originalValue = ((BigDecimal) ((FloatingPointLiteral) expr).getValue()).doubleValue();
                Double newValue = new Double(Math.sin(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            throw new IllegalOperationException(this, "sinMethod", expr);
        }
        else
        {
            return qs.getStoreManager().getDatastoreAdapter().getNumericExpressionForMethod("sin", expr);
        }
    }

    /**
     * Returns the tangent of the argument.
     * @param expr the expression
     * @return the result in a ScalarExpression instance
     */
    public ScalarExpression tanMethod(ScalarExpression expr)
    {
        if (expr == null)
        {
            return new NullLiteral(qs);
        }
        if (expr instanceof Literal)
        {
            if (expr instanceof ByteLiteral)
            {
                int originalValue = ((BigInteger) ((ByteLiteral) expr).getValue()).intValue();
                Double newValue = new Double(Math.tan(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            else if (expr instanceof IntegerLiteral)
            {
                int originalValue = ((Number) ((IntegerLiteral) expr).getValue()).intValue();
                Double newValue = new Double(Math.tan(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            else if (expr instanceof FloatingPointLiteral)
            {
                double originalValue = ((BigDecimal) ((FloatingPointLiteral) expr).getValue()).doubleValue();
                Double newValue = new Double(Math.tan(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            throw new IllegalOperationException(this, "tanMethod", expr);
        }
        else
        {
            return qs.getStoreManager().getDatastoreAdapter().getNumericExpressionForMethod("tan", expr);
        }
    }

    /**
     * Returns the arc cosine of the argument.
     * @param expr the expression
     * @return the result in a ScalarExpression instance
     */
    public ScalarExpression acosMethod(ScalarExpression expr)
    {
        if (expr == null)
        {
            return new NullLiteral(qs);
        }
        if (expr instanceof Literal)
        {
            if (expr instanceof ByteLiteral)
            {
                int originalValue = ((BigInteger) ((ByteLiteral) expr).getValue()).intValue();
                Double newValue = new Double(Math.acos(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            else if (expr instanceof IntegerLiteral)
            {
                int originalValue = ((Number) ((IntegerLiteral) expr).getValue()).intValue();
                Double newValue = new Double(Math.acos(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            else if (expr instanceof FloatingPointLiteral)
            {
                double originalValue = ((BigDecimal) ((FloatingPointLiteral) expr).getValue()).doubleValue();
                Double newValue = new Double(Math.acos(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            throw new IllegalOperationException(this, "acosMethod", expr);
        }
        else
        {
            return qs.getStoreManager().getDatastoreAdapter().getNumericExpressionForMethod("acos", expr);
        }
    }

    /**
     * Returns the arc sine of the argument.
     * @param expr the expression
     * @return the result in a ScalarExpression instance
     */
    public ScalarExpression asinMethod(ScalarExpression expr)
    {
        if (expr == null)
        {
            return new NullLiteral(qs);
        }
        if (expr instanceof Literal)
        {
            if (expr instanceof ByteLiteral)
            {
                int originalValue = ((BigInteger) ((ByteLiteral) expr).getValue()).intValue();
                Double newValue = new Double(Math.asin(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            else if (expr instanceof IntegerLiteral)
            {
                int originalValue = ((Number) ((IntegerLiteral) expr).getValue()).intValue();
                Double newValue = new Double(Math.asin(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            else if (expr instanceof FloatingPointLiteral)
            {
                double originalValue = ((BigDecimal) ((FloatingPointLiteral) expr).getValue()).doubleValue();
                Double newValue = new Double(Math.asin(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            throw new IllegalOperationException(this, "asinMethod", expr);
        }
        else
        {
            return qs.getStoreManager().getDatastoreAdapter().getNumericExpressionForMethod("asin", expr);
        }
    }

    /**
     * Returns the arc tangent of the argument.
     * @param expr the expression
     * @return the result in a ScalarExpression instance
     */
    public ScalarExpression atanMethod(ScalarExpression expr)
    {
        if (expr == null)
        {
            return new NullLiteral(qs);
        }
        if (expr instanceof Literal)
        {
            if (expr instanceof ByteLiteral)
            {
                int originalValue = ((BigInteger) ((ByteLiteral) expr).getValue()).intValue();
                Double newValue = new Double(Math.atan(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            else if (expr instanceof IntegerLiteral)
            {
                int originalValue = ((Number) ((IntegerLiteral) expr).getValue()).intValue();
                Double newValue = new Double(Math.atan(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            else if (expr instanceof FloatingPointLiteral)
            {
                double originalValue = ((BigDecimal) ((FloatingPointLiteral) expr).getValue()).doubleValue();
                Double newValue = new Double(Math.atan(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            throw new IllegalOperationException(this, "atanMethod", expr);
        }
        else
        {
            return qs.getStoreManager().getDatastoreAdapter().getNumericExpressionForMethod("atan", expr);
        }
    }

    /**
     * Returns the exponent of the argument.
     * @param expr the expression
     * @return the result in a ScalarExpression instance
     */
    public ScalarExpression expMethod(ScalarExpression expr)
    {
        if (expr == null)
        {
            return new NullLiteral(qs);
        }
        if (expr instanceof Literal)
        {
            if (expr instanceof ByteLiteral)
            {
                int originalValue = ((BigInteger) ((ByteLiteral) expr).getValue()).intValue();
                Double newValue = new Double(Math.exp(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            else if (expr instanceof IntegerLiteral)
            {
                int originalValue = ((Number) ((IntegerLiteral) expr).getValue()).intValue();
                Double newValue = new Double(Math.exp(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            else if (expr instanceof FloatingPointLiteral)
            {
                double originalValue = ((BigDecimal) ((FloatingPointLiteral) expr).getValue()).doubleValue();
                Double newValue = new Double(Math.exp(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            throw new IllegalOperationException(this, "expMethod", expr);
        }
        else
        {
            return qs.getStoreManager().getDatastoreAdapter().getNumericExpressionForMethod("exp", expr);
        }
    }

    /**
     * Returns the log of the argument.
     * @param expr the expression
     * @return the result in a ScalarExpression instance
     */
    public ScalarExpression logMethod(ScalarExpression expr)
    {
        if (expr == null)
        {
            return new NullLiteral(qs);
        }
        if (expr instanceof Literal)
        {
            if (expr instanceof ByteLiteral)
            {
                int originalValue = ((BigInteger) ((ByteLiteral) expr).getValue()).intValue();
                Double newValue = new Double(Math.log(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            else if (expr instanceof IntegerLiteral)
            {
                int originalValue = ((Number) ((IntegerLiteral) expr).getValue()).intValue();
                Double newValue = new Double(Math.log(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            else if (expr instanceof FloatingPointLiteral)
            {
                double originalValue = ((BigDecimal) ((FloatingPointLiteral) expr).getValue()).doubleValue();
                Double newValue = new Double(Math.log(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            throw new IllegalOperationException(this, "logMethod", expr);
        }
        else
        {
            return qs.getStoreManager().getDatastoreAdapter().getNumericExpressionForMethod("log", expr);
        }
    }

    /**
     * Returns the floor of the argument.
     * @param expr the expression
     * @return the result in a ScalarExpression instance
     */
    public ScalarExpression floorMethod(ScalarExpression expr)
    {
        if (expr == null)
        {
            return new NullLiteral(qs);
        }
        if (expr instanceof Literal)
        {
            if (expr instanceof ByteLiteral)
            {
                int originalValue = ((BigInteger) ((ByteLiteral) expr).getValue()).intValue();
                Double newValue = new Double(Math.floor(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            else if (expr instanceof IntegerLiteral)
            {
                int originalValue = ((Number) ((IntegerLiteral) expr).getValue()).intValue();
                Double newValue = new Double(Math.floor(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            else if (expr instanceof FloatingPointLiteral)
            {
                double originalValue = ((BigDecimal) ((FloatingPointLiteral) expr).getValue()).doubleValue();
                Double newValue = new Double(Math.floor(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            throw new IllegalOperationException(this, "floorMethod", expr);
        }
        else
        {
            return qs.getStoreManager().getDatastoreAdapter().getNumericExpressionForMethod("floor", expr);
        }
    }

    /**
     * Returns the ceiling of the argument.
     * @param expr the expression
     * @return the result in a ScalarExpression instance
     */
    public ScalarExpression ceilMethod(ScalarExpression expr)
    {
        if (expr == null)
        {
            return new NullLiteral(qs);
        }
        if (expr instanceof Literal)
        {
            if (expr instanceof ByteLiteral)
            {
                int originalValue = ((BigInteger) ((ByteLiteral) expr).getValue()).intValue();
                Double newValue = new Double(Math.ceil(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            else if (expr instanceof IntegerLiteral)
            {
                int originalValue = ((Number) ((IntegerLiteral) expr).getValue()).intValue();
                Double newValue = new Double(Math.ceil(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            else if (expr instanceof FloatingPointLiteral)
            {
                double originalValue = ((BigDecimal) ((FloatingPointLiteral) expr).getValue()).doubleValue();
                Double newValue = new Double(Math.ceil(originalValue));
                return new FloatingPointLiteral(qs, expr.mapping, newValue);
            }
            throw new IllegalOperationException(this, "ceilMethod", expr);
        }
        else
        {
            return qs.getStoreManager().getDatastoreAdapter().getNumericExpressionForMethod("ceil", expr);
        }
    }
}