/*
 *  JavaDump 1.0
 *  Copyright (c) 1997 Matt T. Yourst. All rights reserved.
 *
 *  File:     DumpClassToHtml.java
 *  Module:   DumpClassToHtml class
 *  Version:  1.00.053
 *  Date:     May - June 1997
 *  Author:   Matt Yourst [yourst@laserstars.com]
 *  Language: Java 1.1+
 *
 *  Just changed it, so output format is in ascii. 
 *  Chr. Clemens Lee 2000-04-05 clemens@kclee.com
 */

package ccl.jcf;

import java.util.*;
import lti.java.jcf.*;

/**
 * Provides a method for documenting the low-level contents of
 * a class file (represented by a JcfClassFile object) in a
 * standard HTML file.
 */
public class DumpClassToTxt implements RuntimeConstants
{
  /**
   * If specified as part of a bit mask, each flag prevents
   * the dumping of the respective class file component.
   */
  public static final int OMIT_CONSTPOOL = 0x1;
  public static final int OMIT_CLASSDESC = 0x2;
  public static final int OMIT_INTERFACES = 0x4;
  public static final int OMIT_FIELDS = 0x8;
  public static final int OMIT_METHODS = 0x10;
  public static final int OMIT_EXTRAATTRIBUTES = 0x20;
  public static final int OMIT_ALLATTRIBUTES = 0x40;
  
  protected JcfClassFile jcf;
  protected JcfConstantPool cp;
  protected StringBuffer output;
  protected int flags;
   
  /**
   * Creates a new DumpClassToTxt object with the specified source
   * class, output stream, and dumping flags.
   *
   * @param       flags       Created as a bit mask of the
   *                          OMIT_xxx fields of this class.
   *
   * @exception   Exception   don't know why this could be
   *                          thrown. Propably all kinds of
   *                          of ClassNotFoundException etc.
   *                          can be thrown. I think 
   *                          IOException and ZipException are
   *                          other possible exceptions to
   *                          receive. ccl.
   */
  public DumpClassToTxt(JcfClassFile jcf, StringBuffer output, int flags) 
      throws Exception
  {
    super();
    this.jcf = jcf;
    this.cp = jcf.getConstantPool();
    this.output = output;
    this.flags = flags;
  }

  /**
   * Dump the class previously set up at instantiation to the output stream.
   */
  public void dump()
  {
    int i; // Utility variable
    int j; // Utility variable
    int m; // Maximum bound of utility variable
    Enumeration e; // Utility enumeration
    
    Date now = new Date(); // Get time of generation
    //
    // Prepare HTML header
    //
    println("Class Dump: " + jcf.getFullName());
    println("JavaDump 1.0 - Copyright 1997 Matt Yourst");
    println("Dump of class:");
    println("" + jcf.getFullName().replace('/', '.'));
    println("(Generated " + now.toString() + ")");
         println();
    println("Table of Contents");
    if ((flags & OMIT_CONSTPOOL) == 0) println(" - Constant Pool");
    if ((flags & OMIT_CLASSDESC) == 0) println(" - Class Descriptor");
    if ((flags & OMIT_INTERFACES) == 0) println(" - Interfaces");
    if ((flags & OMIT_FIELDS) == 0) println(" - Fields");
    if ((flags & OMIT_METHODS) == 0) println(" - Methods");
    if ((flags & OMIT_EXTRAATTRIBUTES) == 0) println(" - Attributes");
    println();
    //
    // Constant Pool
    //
    if ((flags & OMIT_CONSTPOOL) == 0)
    {
      cp = jcf.getConstantPool();
      printConstantPool();
    }
    //
    // Class Descriptor
    //
    if ((flags & OMIT_CLASSDESC) == 0)
    {
      println("Class Descriptor");
      println("Access flags: 0x" +
      Integer.toHexString(jcf.hdrAccessFlags));
      println("This class: " + cpRef(jcf.hdrThisClassName) +
        " (" + cp.classNameAt(jcf.hdrThisClassName) + ")");
      println("Superclass: " + cpRef(jcf.hdrSuperClassName) +
        " (" + cp.classNameAt(jcf.hdrSuperClassName) + ")");
      println();
    }
    //
    // Interfaces
    //
    if ((flags & OMIT_INTERFACES) == 0)
    {    
      JcfInterfaceCollection interfaces = jcf.getInterfaces();
      m = interfaces.data.length;
      println("Interfaces (" + m +
        ")");
      for (i = 0; i < m; i++)
      {
        j = interfaces.data[i];      
        println(" - " + cpRef(j) + " (" + cp.classNameAt(j) + ")");
      }
      println();
    }
    //
    // Fields
    //
    JcfMemberCollection members;    
    if ((flags & OMIT_FIELDS) == 0)
    {
      members = jcf.getFields();
      printMembers(members, "Fields");
    }
    //
    // Methods
    //
    if ((flags & OMIT_METHODS) == 0)
    {
      members = jcf.getMethods();
      printMembers(members, "Methods");
    }
    //
    // Extra attributes
    //
    if ((flags & OMIT_EXTRAATTRIBUTES) == 0)
    {
      println("Attributes (" + jcf.getAttributes().size() + ")");
      printAttributes(jcf.getAttributes());
    }
    println("\nEnd of report");
    println("Report produced by JavaDump 1.0. Copyright 1997 Matt Yourst.");
  }

  /**
   * Returns an HTML hyperlink string (anchor) that has the text
   * "#123" (123 is in place of [cpr]) and links to the anchor #cp123.
   */
  protected static String cpRef(int cpr)
  {
    return "" + cpr;
  }
  
  /**
   * Print the constant pool (e.g. in a table format.)
   */
  protected void printConstantPool()
  {
    Enumeration e;
    int i;
    println("Constant Pool (" + cp.size() + " slots)");
    println();
    println("Index Type        Data");
    e = cp.elements();
    i = 0;
    while (e.hasMoreElements())
    {
      CptGeneric cpEntry = (CptGeneric)e.nextElement();
      if (cpEntry == null)
      {
        i++;
        continue; // Nulls are possible after 64-bit (2 slot) tags
      }
      print("" + i++ + "     ");
      switch (cpEntry.getTag())
      {
        case CONSTANT_UTF8:
          print("UTF-8       \"" + ((CptUTF)cpEntry).value + "\" (" +
            ((CptUTF)cpEntry).value.length() + " chars)"); break;
        case CONSTANT_NAMEANDTYPE:
          print("NameType    name " + cpRef(((CptNameType)cpEntry).nameIndex) +
          " type " + cpRef(((CptNameType)cpEntry).signatureIndex)); break;
        case CONSTANT_CLASS:
          print("Class       name " + cpRef(((CptClass)cpEntry).nameIndex));
          break;
        case CONSTANT_METHOD:
          print("Method      class " +
          cpRef(((CptMemberOrInterface)cpEntry).classIndex) +
          " type " + cpRef(((CptMemberOrInterface)cpEntry).nameTypeIndex)); break;
        case CONSTANT_FIELD:
          print("Field       class " +
          cpRef(((CptMemberOrInterface)cpEntry).classIndex) +
          " type " + cpRef(((CptMemberOrInterface)cpEntry).nameTypeIndex)); break;
        case CONSTANT_STRING:
          print("String      data " + cpRef(((CptString)cpEntry).StringIndex));
          break;
        case CONSTANT_INTEGER:
          print("Integer     " + ((CptInteger)cpEntry).value);
          break;
        case CONSTANT_FLOAT:
          print("Float       " + ((CptFloat)cpEntry).value);
          break;
        case CONSTANT_LONG:
          print("Long        " + ((CptLong)cpEntry).value);
          break;
        case CONSTANT_DOUBLE:
          print("Double      " + ((CptDouble)cpEntry).value);
          break;
        case CONSTANT_INTERFACEMETHOD:
          print("IntMethod   class " +
          cpRef(((CptMemberOrInterface)cpEntry).classIndex) +
          " type " + cpRef(((CptMemberOrInterface)cpEntry).nameTypeIndex)); break;
        case CONSTANT_UNICODE:
          print("Unicode     " + ((CptUnicode)cpEntry).value.length + " bytes");
          break;
        default:
          print("???         ???");
          break;
      }
      println();
    }
    println();
  }
  
  /**
   * Print descriptions of all members in the collection.
   * @param fieldOrMethod the text ("Fields" or "Methods") to print as a heading.
   */
  protected void printMembers(JcfMemberCollection members, String fieldOrMethod)
  {
    Enumeration e = members.elements();
    println("" + fieldOrMethod + " (" +
      members.size() + ")");
    while (e.hasMoreElements())
    {
      JcfMember member = (JcfMember)e.nextElement();
      println(" - " + cp.utfAt(member.mbrNameIndex) + " (" +
                                  cpRef(member.mbrNameIndex) + ") signature \"" +
                                  cp.utfAt(member.mbrSignatureIndex) + "\" (" +
                                  cpRef(member.mbrSignatureIndex) + ") access 0x" +
                                  Integer.toHexString(member.mbrAccessFlags));
      printAttributes(member.getAttributes());
    }
    println();
  }

  /**
   * Print an attribute collection. Should recurse through nested
   * attributes using appropriate indentation.
   */
  protected void printAttributes(JcfAttributeCollection attrs)
  {
    if ((flags & OMIT_ALLATTRIBUTES) != 0) return;
    Enumeration e = attrs.elements();
    while (e.hasMoreElements())
    {
      byte[] data;
      JcfAttribute attr = (JcfAttribute)e.nextElement();
      String attrname = cp.utfAt(attr.getNameIndex());
      println("   . \"" + attrname + "\" (" +
        cpRef(attr.getNameIndex()) + "): " + attr.getLength() + " bytes");
      if (attrname.equals("Code"))
      {
                        JcfCodeAttribute codeattr = (JcfCodeAttribute)attr;
                        println("     [stack words: " + codeattr.catMaxStack +
                                          ", local words: " + codeattr.catMaxLocals +
                                          ", bytecode: " + codeattr.catCode.length + " bytes]");
                        printAttributes(codeattr.catExtraAttributes);
      }
 else if (attrname.equals("ConstantValue")) 
{
                        data = attr.getData();
                        println("     [constant: " + cpRef((data[0] << 8) | data[1]) + "]");
      }
 else if (attrname.equals("SourceFile")) 
{
                        data = attr.getData();
                        int idx = ((data[0] & 0xff) << 8) | (data[1] & 0xff);
                        println("     [filename: " + cpRef(idx) + " = " + cp.utfAt(idx) + "]");
      }
    }
  }

  /**
   * Print a line. Convenience method.
   */
  protected final void println(String s)
  {
    output.append(s).append('\n');
  }
  /**
   * Print a text fragment. Convenience method.
   */
  protected final void print(String s)
  {
    output.append(s);
  }
  /**
   * Print an empty line. Convenience method.
   */
  protected final void println()
  {
    output.append('\n');
  }
}
