/*
 * Decompiled with CFR 0.152.
 */
package ghidra.server.store;

import db.buffers.LocalManagedBufferFile;
import ghidra.framework.store.FileSystem;
import ghidra.framework.store.local.LocalFileSystem;
import ghidra.server.Repository;
import ghidra.server.RepositoryManager;
import ghidra.server.store.RepositoryFile;
import ghidra.util.InvalidNameException;
import ghidra.util.exception.DuplicateFileException;
import ghidra.util.exception.FileInUseException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class RepositoryFolder {
    static final Logger log = LogManager.getLogger(RepositoryFolder.class);
    private Repository repository;
    private LocalFileSystem fileSystem;
    private RepositoryFolder parent;
    private String name;
    private Map<String, RepositoryFolder> folderMap = new HashMap<String, RepositoryFolder>();
    private Map<String, RepositoryFile> fileMap = new HashMap<String, RepositoryFile>();
    private static Comparator<RepositoryFile> FILE_NAME_COMPARATOR = (f1, f2) -> f1.getName().compareTo(f2.getName());
    private static Comparator<RepositoryFolder> FOLDER_NAME_COMPARATOR = (f1, f2) -> f1.getName().compareTo(f2.getName());

    private RepositoryFolder(Repository repository, LocalFileSystem fileSystem, RepositoryFolder parent, String name) throws IOException {
        this.repository = repository;
        this.fileSystem = fileSystem;
        this.parent = parent;
        this.name = name;
        this.init();
    }

    public RepositoryFolder(Repository repository, LocalFileSystem fileSystem) throws IOException {
        this.repository = repository;
        this.fileSystem = fileSystem;
        this.name = "";
        this.init();
    }

    private void init() throws IOException {
        int i;
        String path = this.getPathname();
        String[] names = this.fileSystem.getFolderNames(path);
        for (i = 0; i < names.length; ++i) {
            RepositoryFolder subfolder = new RepositoryFolder(this.repository, this.fileSystem, this, names[i]);
            this.folderMap.put(names[i], subfolder);
        }
        names = this.fileSystem.getItemNames(path);
        for (i = 0; i < names.length; ++i) {
            try {
                RepositoryFile rf = new RepositoryFile(this.repository, this.fileSystem, this, names[i]);
                this.fileMap.put(names[i], rf);
                continue;
            }
            catch (FileNotFoundException fileNotFoundException) {
                // empty catch block
            }
        }
    }

    public String getName() {
        return this.name;
    }

    public RepositoryFolder getParent() {
        return this.parent;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String getPathname() {
        LocalFileSystem localFileSystem = this.fileSystem;
        synchronized (localFileSystem) {
            StringBuffer buf = new StringBuffer(this.parent != null ? this.parent.getPathname() : "");
            if (buf.length() != 1) {
                buf.append('/');
            }
            buf.append(this.name);
            return buf.toString();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public RepositoryFolder[] getFolders() {
        LocalFileSystem localFileSystem = this.fileSystem;
        synchronized (localFileSystem) {
            RepositoryFolder[] folders = new RepositoryFolder[this.folderMap.size()];
            folders = this.folderMap.values().toArray(folders);
            Arrays.sort(folders, FOLDER_NAME_COMPARATOR);
            return folders;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public RepositoryFolder getFolder(String folderName) {
        LocalFileSystem localFileSystem = this.fileSystem;
        synchronized (localFileSystem) {
            RepositoryFolder rf = this.folderMap.get(folderName);
            if (rf != null) {
                return rf;
            }
            String path = RepositoryFolder.makePathname(this.getPathname(), folderName);
            if (this.fileSystem.folderExists(path)) {
                try {
                    rf = new RepositoryFolder(this.repository, this.fileSystem, this, folderName);
                    this.folderMap.put(folderName, rf);
                    return rf;
                }
                catch (IOException e) {
                    log.error("Repository error: " + this.repository.getName() + ": " + e.getMessage());
                }
            }
            return null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public RepositoryFile[] getFiles() {
        LocalFileSystem localFileSystem = this.fileSystem;
        synchronized (localFileSystem) {
            RepositoryFile[] files = new RepositoryFile[this.fileMap.size()];
            files = this.fileMap.values().toArray(files);
            Arrays.sort(files, FILE_NAME_COMPARATOR);
            return files;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public RepositoryFile getFile(String fileName) {
        LocalFileSystem localFileSystem = this.fileSystem;
        synchronized (localFileSystem) {
            RepositoryFile rf = this.fileMap.get(fileName);
            if (rf != null) {
                return rf;
            }
            if (this.fileSystem.fileExists(this.getPathname(), fileName)) {
                try {
                    rf = new RepositoryFile(this.repository, this.fileSystem, this, fileName);
                    this.fileMap.put(fileName, rf);
                    return rf;
                }
                catch (IOException e) {
                    log.error("Repository '" + this.repository.getName() + "' file error: " + e.getMessage() + "\n    " + RepositoryFolder.makePathname(this.getPathname(), fileName));
                }
            }
            return null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public RepositoryFolder createFolder(String folderName, String user) throws InvalidNameException, IOException {
        LocalFileSystem localFileSystem = this.fileSystem;
        synchronized (localFileSystem) {
            this.repository.validate();
            this.repository.validateWritePrivilege(user);
            if (this.getFolder(folderName) != null) {
                throw new DuplicateFileException(folderName + " already exists");
            }
            this.fileSystem.createFolder(this.getPathname(), folderName);
            RepositoryFolder rf = this.getFolder(folderName);
            RepositoryManager.log(this.repository.getName(), rf.getPathname(), "folder created", user);
            return rf;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public LocalManagedBufferFile createDatabase(String itemName, String fileID, int bufferSize, String contentType, String user, String projectPath) throws InvalidNameException, IOException {
        LocalFileSystem localFileSystem = this.fileSystem;
        synchronized (localFileSystem) {
            this.repository.validate();
            this.repository.validateWritePrivilege(user);
            if (this.getFile(itemName) != null) {
                throw new DuplicateFileException(itemName + " already exists");
            }
            LocalManagedBufferFile bf = this.fileSystem.createDatabase(this.getPathname(), itemName, fileID, contentType, bufferSize, user, projectPath);
            RepositoryManager.log(this.repository.getName(), RepositoryFolder.makePathname(this.getPathname(), itemName), "file created", user);
            return bf;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void delete() throws IOException {
        LocalFileSystem localFileSystem = this.fileSystem;
        synchronized (localFileSystem) {
            this.repository.validate();
            if (this.parent == null) {
                throw new IOException("Root folder may not be deleted");
            }
            this.fileSystem.deleteFolder(this.getPathname());
            this.parent.folderMap.remove(this.name);
        }
    }

    private boolean containsCheckout() throws IOException {
        for (RepositoryFile rf : this.fileMap.values()) {
            if (!rf.hasCheckouts()) continue;
            return true;
        }
        for (RepositoryFolder subfolder : this.folderMap.values()) {
            if (!subfolder.containsCheckout()) continue;
            return true;
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void fileDeleted(RepositoryFile rf) {
        LocalFileSystem localFileSystem = this.fileSystem;
        synchronized (localFileSystem) {
            this.fileMap.remove(rf.getName());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void fileMoved(RepositoryFile rf, String oldName, RepositoryFolder newFolder) {
        LocalFileSystem localFileSystem = this.fileSystem;
        synchronized (localFileSystem) {
            this.fileMap.remove(oldName);
            newFolder.fileMap.put(rf.getName(), rf);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void moveTo(RepositoryFolder newParent, String newFolderName, String user) throws InvalidNameException, IOException {
        LocalFileSystem localFileSystem = this.fileSystem;
        synchronized (localFileSystem) {
            this.repository.suspendEventDispatching();
            try {
                this.repository.validate();
                this.repository.validateWritePrivilege(user);
                if (this.parent == null) {
                    throw new IOException("Root folder may not be moved");
                }
                if (newParent.isDescendentOf(this)) {
                    throw new IOException("New folder must not be decendent");
                }
                if (this.containsCheckout()) {
                    throw new FileInUseException(this.getPathname() + " contains one or more checked-out files");
                }
                String oldPath = this.getPathname();
                if (this.name.equals(newFolderName)) {
                    this.fileSystem.moveFolder(this.parent.getPathname(), this.name, newParent.getPathname());
                    this.parent.folderMap.remove(newFolderName);
                    newParent.folderMap.put(newFolderName, this);
                    this.parent = newParent;
                } else if (this.parent.equals(newParent)) {
                    this.fileSystem.renameFolder(this.parent.getPathname(), this.name, newFolderName);
                    this.parent.folderMap.remove(this.name);
                    this.name = newFolderName;
                    this.parent.folderMap.put(newFolderName, this);
                } else {
                    throw new IOException("Folder can not be renamed and moved");
                }
                this.pathChanged();
                RepositoryManager.log(this.repository.getName(), oldPath, "folder moved to " + this.getPathname(), user);
            }
            finally {
                this.repository.flushChangeEvents();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void pathChanged() {
        LocalFileSystem localFileSystem = this.fileSystem;
        synchronized (localFileSystem) {
            for (RepositoryFile rf : this.fileMap.values()) {
                rf.pathChanged();
            }
            for (RepositoryFolder subfolder : this.folderMap.values()) {
                subfolder.pathChanged();
            }
        }
    }

    private boolean isDescendentOf(RepositoryFolder folder) {
        RepositoryFolder rf = this.parent;
        while (rf != null) {
            if (rf == folder) {
                return true;
            }
            rf = rf.parent;
        }
        return false;
    }

    public static String makePathname(String parentPath, String childName) {
        String path = parentPath.endsWith(FileSystem.SEPARATOR) ? parentPath.substring(0, parentPath.length() - FileSystem.SEPARATOR.length()) : parentPath;
        return path + FileSystem.SEPARATOR + childName;
    }
}

