/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.plugin.core.analysis;

import generic.jar.ResourceFile;
import ghidra.app.plugin.core.analysis.NonReturningFunctionNames;
import ghidra.app.services.AbstractAnalyzer;
import ghidra.app.services.AnalysisPriority;
import ghidra.app.services.AnalyzerType;
import ghidra.app.util.MemoryBlockUtils;
import ghidra.app.util.bin.format.dwarf4.DWARFUtil;
import ghidra.app.util.bin.format.elf.info.ElfInfoItem;
import ghidra.app.util.bin.format.golang.GoBuildInfo;
import ghidra.app.util.bin.format.golang.GoFunctionFixup;
import ghidra.app.util.bin.format.golang.GoRegisterInfo;
import ghidra.app.util.bin.format.golang.GoVer;
import ghidra.app.util.bin.format.golang.PEGoBuildId;
import ghidra.app.util.bin.format.golang.rtti.GoFuncData;
import ghidra.app.util.bin.format.golang.rtti.GoModuledata;
import ghidra.app.util.bin.format.golang.rtti.GoRttiMapper;
import ghidra.app.util.bin.format.golang.structmapping.MarkupSession;
import ghidra.app.util.importer.MessageLog;
import ghidra.framework.options.Options;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressSetView;
import ghidra.program.model.data.AbstractUnsignedIntegerDataType;
import ghidra.program.model.data.DataType;
import ghidra.program.model.data.Pointer;
import ghidra.program.model.data.Structure;
import ghidra.program.model.data.VoidDataType;
import ghidra.program.model.lang.PrototypeModel;
import ghidra.program.model.lang.Register;
import ghidra.program.model.listing.ContextChangeException;
import ghidra.program.model.listing.Function;
import ghidra.program.model.listing.ParameterImpl;
import ghidra.program.model.listing.Program;
import ghidra.program.model.listing.ReturnParameterImpl;
import ghidra.program.model.listing.Variable;
import ghidra.program.model.mem.MemoryBlock;
import ghidra.program.model.symbol.SourceType;
import ghidra.program.model.symbol.Symbol;
import ghidra.program.model.symbol.SymbolTable;
import ghidra.program.model.symbol.SymbolType;
import ghidra.program.model.symbol.SymbolUtilities;
import ghidra.util.Msg;
import ghidra.util.NumericUtilities;
import ghidra.util.exception.CancelledException;
import ghidra.util.exception.DuplicateNameException;
import ghidra.util.exception.InvalidInputException;
import ghidra.util.task.TaskMonitor;
import ghidra.util.task.UnknownProgressWrappingTaskMonitor;
import ghidra.xml.XmlParseException;
import java.io.File;
import java.io.IOException;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import utilities.util.FileUtilities;

public class GolangSymbolAnalyzer
extends AbstractAnalyzer {
    private static final String NAME = "Golang Symbol";
    private static final String DESCRIPTION = "Analyze Golang binaries for RTTI and function symbols.\n'Apply Data Archives' and 'Shared Return Calls' analyzers should be disabled for best results.";
    private static final String ARTIFICIAL_RUNTIME_ZEROBASE_SYMBOLNAME = "ARTIFICIAL.runtime.zerobase";
    private GolangAnalyzerOptions analyzerOptions = new GolangAnalyzerOptions();

    public GolangSymbolAnalyzer() {
        super(NAME, DESCRIPTION, AnalyzerType.BYTE_ANALYZER);
        this.setPriority(AnalysisPriority.FORMAT_ANALYSIS.after().after());
        this.setDefaultEnablement(true);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public boolean added(Program program, AddressSetView set, TaskMonitor monitor, MessageLog log) throws CancelledException {
        monitor.setMessage("Golang symbol analyzer");
        try (GoRttiMapper goBinary = GoRttiMapper.getMapperFor(program, log);){
            if (goBinary == null) {
                Msg.error((Object)this, (Object)"Golang analyzer error: unable to get GoRttiMapper");
                boolean bl = false;
                return bl;
            }
            goBinary.init(monitor);
            goBinary.discoverGoTypes(monitor);
            UnknownProgressWrappingTaskMonitor upwtm = new UnknownProgressWrappingTaskMonitor(monitor, 100L);
            upwtm.initialize(0L);
            upwtm.setMessage("Marking up Golang RTTI structures");
            MarkupSession markupSession = goBinary.createMarkupSession((TaskMonitor)upwtm);
            GoModuledata firstModule = goBinary.getFirstModule();
            if (firstModule != null) {
                markupSession.labelStructure(firstModule, "firstmoduledata");
                markupSession.markup(firstModule, false);
            }
            this.markupWellknownSymbols(goBinary, markupSession);
            this.setupProgramContext(goBinary, markupSession);
            goBinary.recoverDataTypes(monitor);
            this.markupGoFunctions(goBinary, markupSession);
            this.fixupNoReturnFuncs(program);
            this.markupMiscInfoStructs(program);
            if (!this.analyzerOptions.createBootstrapDatatypeArchive) return true;
            this.createBootstrapGDT(goBinary, program, monitor);
            return true;
        }
        catch (IOException e) {
            Msg.error((Object)this, (Object)"Golang analysis failure", (Throwable)e);
        }
        return true;
    }

    @Override
    public void registerOptions(Options options, Program program) {
        options.registerOption("Create Bootstrap GDT", (Object)this.analyzerOptions.createBootstrapDatatypeArchive, null, "Creates a Ghidra data type archive that contains just the necessary data types to parse other golang binaries. DWARF data is needed for this to succeed. The new GDT file will be placed in the user's home directory and will be called golang_MajorVer.MinorVer_XXbit_osname.NNNNNNNNN.gdt, where NNNNNN is a timestamp.");
    }

    @Override
    public void optionsChanged(Options options, Program program) {
        this.analyzerOptions.createBootstrapDatatypeArchive = options.getBoolean("Create Bootstrap GDT", this.analyzerOptions.createBootstrapDatatypeArchive);
    }

    private void markupWellknownSymbols(GoRttiMapper goBinary, MarkupSession session) throws IOException {
        Program program = goBinary.getProgram();
        Symbol g0 = SymbolUtilities.getUniqueSymbol((Program)program, (String)"runtime.g0");
        Structure gStruct = goBinary.getGhidraDataType("runtime.g", Structure.class);
        if (g0 != null && gStruct != null) {
            session.markupAddressIfUndefined(g0.getAddress(), (DataType)gStruct);
        }
        Symbol m0 = SymbolUtilities.getUniqueSymbol((Program)program, (String)"runtime.m0");
        Structure mStruct = goBinary.getGhidraDataType("runtime.m", Structure.class);
        if (m0 != null && mStruct != null) {
            session.markupAddressIfUndefined(m0.getAddress(), (DataType)mStruct);
        }
    }

    private void markupGoFunctions(GoRttiMapper goBinary, MarkupSession markupSession) throws IOException {
        for (GoFuncData funcdata : goBinary.getAllFunctions()) {
            String funcname = SymbolUtilities.replaceInvalidChars((String)funcdata.getName(), (boolean)true);
            markupSession.createFunctionIfMissing(funcname, funcdata.getFuncAddress());
        }
        try {
            this.fixDuffFunctions(goBinary, markupSession);
        }
        catch (DuplicateNameException | InvalidInputException e) {
            Msg.error((Object)this, (Object)"Error configuring duff functions", (Throwable)e);
        }
    }

    private void fixDuffFunctions(GoRttiMapper goBinary, MarkupSession session) throws InvalidInputException, DuplicateNameException {
        GoFuncData duffcopyFuncdata;
        Program program = goBinary.getProgram();
        GoRegisterInfo regInfo = goBinary.getRegInfo();
        Pointer voidPtr = program.getDataTypeManager().getPointer((DataType)VoidDataType.dataType);
        DataType uintDT = goBinary.getTypeOrDefault("uint", DataType.class, AbstractUnsignedIntegerDataType.getUnsignedDataType((int)goBinary.getPtrSize(), null));
        GoFuncData duffzeroFuncdata = goBinary.getFunctionByName("runtime.duffzero");
        Function duffzeroFunc = duffzeroFuncdata != null ? program.getFunctionManager().getFunctionAt(duffzeroFuncdata.getFuncAddress()) : null;
        PrototypeModel duffzeroCC = goBinary.getDuffzeroCallingConvention();
        if (duffzeroFunc != null && duffzeroCC != null) {
            boolean needZeroValueParam = regInfo.getZeroRegister() == null;
            ArrayList<ParameterImpl> params = new ArrayList<ParameterImpl>();
            params.add(new ParameterImpl("dest", (DataType)voidPtr, program));
            if (needZeroValueParam) {
                params.add(new ParameterImpl("zeroValue", uintDT, program));
            }
            duffzeroFunc.updateFunction(duffzeroCC.getName(), (Variable)new ReturnParameterImpl((DataType)VoidDataType.dataType, program), params, Function.FunctionUpdateType.DYNAMIC_STORAGE_ALL_PARAMS, true, SourceType.ANALYSIS);
            DWARFUtil.appendComment(program, duffzeroFunc.getEntryPoint(), 3, "Golang special function: ", "duffzero", "\n");
        }
        Function duffcopyFunc = (duffcopyFuncdata = goBinary.getFunctionByName("runtime.duffcopy")) != null ? program.getFunctionManager().getFunctionAt(duffcopyFuncdata.getFuncAddress()) : null;
        PrototypeModel duffcopyCC = goBinary.getDuffcopyCallingConvention();
        if (duffcopyFuncdata != null && duffcopyCC != null) {
            List<ParameterImpl> params = List.of(new ParameterImpl("dest", (DataType)voidPtr, program), new ParameterImpl("src", (DataType)voidPtr, program));
            duffcopyFunc.updateFunction(duffcopyCC.getName(), (Variable)new ReturnParameterImpl((DataType)VoidDataType.dataType, program), params, Function.FunctionUpdateType.DYNAMIC_STORAGE_ALL_PARAMS, true, SourceType.ANALYSIS);
            DWARFUtil.appendComment(program, duffcopyFunc.getEntryPoint(), 3, "Golang special function: ", "duffcopy", "\n");
        }
    }

    private void markupMiscInfoStructs(Program program) {
        ElfInfoItem.ItemWithAddress<PEGoBuildId> wrappedPeBuildId;
        ElfInfoItem.ItemWithAddress<GoBuildInfo> wrappedBuildInfo = GoBuildInfo.findBuildInfo(program);
        if (wrappedBuildInfo != null && program.getListing().isUndefined(wrappedBuildInfo.address(), wrappedBuildInfo.address())) {
            wrappedBuildInfo.item().markupProgram(program, wrappedBuildInfo.address());
        }
        if ((wrappedPeBuildId = PEGoBuildId.findBuildId(program)) != null && program.getListing().isUndefined(wrappedPeBuildId.address(), wrappedPeBuildId.address())) {
            Symbol[] buildIdSymbols;
            for (Symbol sym : buildIdSymbols = program.getSymbolTable().getSymbols(wrappedPeBuildId.address())) {
                if (sym.getSymbolType() != SymbolType.FUNCTION) continue;
                String symName = sym.getName();
                sym.delete();
                try {
                    program.getSymbolTable().createLabel(wrappedPeBuildId.address(), symName, SourceType.IMPORTED);
                }
                catch (InvalidInputException invalidInputException) {}
                break;
            }
            wrappedPeBuildId.item().markupProgram(program, wrappedPeBuildId.address());
        }
    }

    private void fixupNoReturnFuncs(Program program) {
        HashSet noreturnFuncnames = new HashSet();
        try {
            for (ResourceFile file : NonReturningFunctionNames.findDataFiles(program)) {
                FileUtilities.getLines((ResourceFile)file).stream().map(String::trim).filter(s -> !s.isBlank() && !s.startsWith("#")).forEach(noreturnFuncnames::add);
            }
        }
        catch (XmlParseException | IOException e) {
            Msg.error((Object)this, (Object)"Failed to read Golang noreturn func data file", (Throwable)e);
        }
        int count = 0;
        SymbolTable symbolTable = program.getSymbolTable();
        for (Symbol symbol : symbolTable.getPrimarySymbolIterator(true)) {
            Function functionAt;
            String name = symbol.getName(false);
            if (symbol.isExternal() || !noreturnFuncnames.contains(name) || (functionAt = program.getFunctionManager().getFunctionAt(symbol.getAddress())) == null || functionAt.hasNoReturn()) continue;
            functionAt.setNoReturn(true);
            program.getBookmarkManager().setBookmark(symbol.getAddress(), "Analysis", "Non-Returning Function", "Non-Returning Golang Function Identified");
            ++count;
        }
        Msg.info((Object)this, (Object)"Marked %d golang funcs as NoReturn".formatted(count));
    }

    private Address createFakeContextMemory(Program program, long len) {
        long offset_from_eom = 0x100000L;
        Address max = program.getAddressFactory().getDefaultAddressSpace().getMaxAddress();
        Address mbStart = max.subtract(offset_from_eom + len - 1L);
        MemoryBlock newMB = MemoryBlockUtils.createUninitializedBlock(program, false, "ARTIFICAL_GOLANG_CONTEXT", mbStart, len, "Artifical memory block created to hold golang context data types", null, true, true, false, null);
        return newMB.getStart();
    }

    private void setupProgramContext(GoRttiMapper goBinary, MarkupSession session) throws IOException {
        Address contextMemoryAddr;
        Program program = goBinary.getProgram();
        GoRegisterInfo goRegInfo = goBinary.getRegInfo();
        MemoryBlock txtMemblock = program.getMemory().getBlock(".text");
        if (txtMemblock != null && goRegInfo.getZeroRegister() != null && !goRegInfo.isZeroRegisterIsBuiltin()) {
            try {
                program.getProgramContext().setValue(goRegInfo.getZeroRegister(), txtMemblock.getStart(), txtMemblock.getEnd(), BigInteger.ZERO);
            }
            catch (ContextChangeException e) {
                Msg.error((Object)this, (Object)"Unexpected Error", (Throwable)e);
            }
        }
        int alignment = goBinary.getPtrSize();
        long sizeNeeded = 0L;
        Symbol zerobase = SymbolUtilities.getUniqueSymbol((Program)program, (String)"runtime.zerobase");
        long zerobaseSymbol = sizeNeeded;
        long gStructOffset = sizeNeeded += zerobase == null ? NumericUtilities.getUnsignedAlignedValue((long)1L, (long)alignment) : 0L;
        Structure gStruct = goBinary.getGhidraDataType("runtime.g", Structure.class);
        long mStructOffset = sizeNeeded += gStruct != null ? NumericUtilities.getUnsignedAlignedValue((long)gStruct.getLength(), (long)alignment) : 0L;
        Structure mStruct = goBinary.getGhidraDataType("runtime.m", Structure.class);
        Address address = contextMemoryAddr = (sizeNeeded += mStruct != null ? NumericUtilities.getUnsignedAlignedValue((long)mStruct.getLength(), (long)alignment) : 0L) > 0L ? this.createFakeContextMemory(program, sizeNeeded) : null;
        if (zerobase == null) {
            session.labelAddress(contextMemoryAddr.add(zerobaseSymbol), ARTIFICIAL_RUNTIME_ZEROBASE_SYMBOLNAME);
        }
        if (gStruct != null) {
            Address gAddr = contextMemoryAddr.add(gStructOffset);
            session.markupAddressIfUndefined(gAddr, (DataType)gStruct);
            session.labelAddress(gAddr, "CURRENT_G");
            Register currentGoroutineReg = goRegInfo.getCurrentGoroutineRegister();
            if (currentGoroutineReg != null && txtMemblock != null) {
                try {
                    program.getProgramContext().setValue(currentGoroutineReg, txtMemblock.getStart(), txtMemblock.getEnd(), gAddr.getOffsetAsBigInteger());
                }
                catch (ContextChangeException e) {
                    Msg.error((Object)this, (Object)"Unexpected Error", (Throwable)e);
                }
            }
        }
        if (mStruct != null) {
            Address mAddr = contextMemoryAddr.add(mStructOffset);
            session.markupAddressIfUndefined(mAddr, (DataType)mStruct);
        }
    }

    private void createBootstrapGDT(GoRttiMapper goBinary, Program program, TaskMonitor monitor) throws IOException {
        GoVer goVer = goBinary.getGolangVersion();
        String osName = GoRttiMapper.getGolangOSString(program);
        String gdtFilename = GoRttiMapper.getGDTFilename(goVer, goBinary.getPtrSize(), osName);
        gdtFilename = gdtFilename.replace(".gdt", "_%d.gdt".formatted(System.currentTimeMillis()));
        File gdt = new File(System.getProperty("user.home"), gdtFilename);
        goBinary.exportTypesToGDT(gdt, monitor);
        Msg.info((Object)this, (Object)("Golang bootstrap GDT created: " + gdt));
    }

    @Override
    public void analysisEnded(Program program) {
    }

    @Override
    public boolean canAnalyze(Program program) {
        return "golang".equals(program.getCompilerSpec().getCompilerSpecDescription().getCompilerSpecName());
    }

    @Override
    public boolean getDefaultEnablement(Program program) {
        return true;
    }

    private static Address getArtificalZerobaseAddress(Program program) {
        Symbol zerobaseSym = SymbolUtilities.getUniqueSymbol((Program)program, (String)ARTIFICIAL_RUNTIME_ZEROBASE_SYMBOLNAME);
        return zerobaseSym != null ? zerobaseSym.getAddress() : null;
    }

    public static Address getZerobaseAddress(Program prog) {
        Address zerobaseAddr;
        Symbol zerobaseSym = SymbolUtilities.getUniqueSymbol((Program)prog, (String)"runtime.zerobase");
        Address address = zerobaseAddr = zerobaseSym != null ? zerobaseSym.getAddress() : GolangSymbolAnalyzer.getArtificalZerobaseAddress(prog);
        if (zerobaseAddr == null) {
            zerobaseAddr = prog.getImageBase().getAddressSpace().getMinAddress();
            Msg.warn(GoFunctionFixup.class, (Object)("Unable to find Golang runtime.zerobase, using " + zerobaseAddr));
        }
        return zerobaseAddr;
    }

    private static class GolangAnalyzerOptions {
        static final String CREATE_BOOTSTRAP_GDT_OPTIONNAME = "Create Bootstrap GDT";
        static final String CREATE_BOOTSTRAP_GDT_DESC = "Creates a Ghidra data type archive that contains just the necessary data types to parse other golang binaries. DWARF data is needed for this to succeed. The new GDT file will be placed in the user's home directory and will be called golang_MajorVer.MinorVer_XXbit_osname.NNNNNNNNN.gdt, where NNNNNN is a timestamp.";
        boolean createBootstrapDatatypeArchive;

        private GolangAnalyzerOptions() {
        }
    }
}

