/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.plugin.core.decompile.actions;

import docking.widgets.EventTrigger;
import ghidra.app.plugin.core.decompile.actions.ASTGraphDisplayListener;
import ghidra.app.services.GraphDisplayBroker;
import ghidra.framework.plugintool.PluginTool;
import ghidra.program.model.address.Address;
import ghidra.program.model.lang.Register;
import ghidra.program.model.listing.Program;
import ghidra.program.model.pcode.HighFunction;
import ghidra.program.model.pcode.HighVariable;
import ghidra.program.model.pcode.PcodeBlock;
import ghidra.program.model.pcode.PcodeBlockBasic;
import ghidra.program.model.pcode.PcodeOp;
import ghidra.program.model.pcode.PcodeOpAST;
import ghidra.program.model.pcode.Varnode;
import ghidra.service.graph.AttributedGraph;
import ghidra.service.graph.AttributedVertex;
import ghidra.service.graph.GraphDisplay;
import ghidra.service.graph.GraphDisplayListener;
import ghidra.util.Msg;
import ghidra.util.NumericUtilities;
import ghidra.util.exception.CancelledException;
import ghidra.util.exception.GraphException;
import ghidra.util.task.Task;
import ghidra.util.task.TaskMonitor;
import java.util.Iterator;

public class ASTGraphTask
extends Task {
    private static final String CODE_ATTRIBUTE = "Code";
    private static final String SYMBOLS_ATTRIBUTE = "Symbols";
    private static final String VERTEX_TYPE_ATTRIBUTE = "VertexType";
    private static final String ENTRY_NODE = "Entry";
    private static final String BODY_NODE = "Body";
    private static final String EXIT_NODE = "Exit";
    private static final String SWITCH_NODE = "Switch";
    private static final String BAD_NODE = "Bad";
    private static final String DATA_NODE = "Data";
    private GraphDisplayBroker graphService;
    private boolean newGraph;
    private int codeLimitPerBlock;
    private Address location;
    private HighFunction hfunction;
    private GraphType graphType;
    private int uniqueNum = 0;
    private PluginTool tool;

    public ASTGraphTask(GraphDisplayBroker graphService, boolean newGraph, int codeLimitPerBlock, Address location, HighFunction hfunction, GraphType graphType, PluginTool tool) {
        super("Graph " + graphType.getName(), true, false, true);
        this.graphService = graphService;
        this.newGraph = newGraph;
        this.codeLimitPerBlock = codeLimitPerBlock;
        this.location = location;
        this.hfunction = hfunction;
        this.graphType = graphType;
        this.tool = tool;
    }

    public void run(TaskMonitor monitor) {
        AttributedGraph graph = new AttributedGraph();
        try {
            monitor.setMessage("Computing Graph...");
            if (this.graphType == GraphType.DATA_FLOW_GRAPH) {
                this.createDataFlowGraph(graph, monitor);
            } else {
                this.createControlFlowGraph(graph, monitor);
            }
            GraphDisplay display = this.graphService.getDefaultGraphDisplay(!this.newGraph, monitor);
            ASTGraphDisplayListener displayListener = new ASTGraphDisplayListener(this.tool, display, this.hfunction, this.graphType);
            display.setGraphDisplayListener((GraphDisplayListener)displayListener);
            monitor.setMessage("Obtaining handle to graph provider...");
            if (monitor.isCancelled()) {
                return;
            }
            monitor.setCancelEnabled(false);
            monitor.setMessage("Rendering Graph...");
            display.defineVertexAttribute(CODE_ATTRIBUTE);
            display.defineVertexAttribute(SYMBOLS_ATTRIBUTE);
            display.setVertexLabel(CODE_ATTRIBUTE, 0, 12, true, this.graphType == GraphType.CONTROL_FLOW_GRAPH ? this.codeLimitPerBlock + 1 : 1);
            Object description = this.graphType == GraphType.DATA_FLOW_GRAPH ? "AST Data Flow" : "AST Control Flow";
            description = (String)description + " for " + this.hfunction.getFunction().getName();
            display.setGraph(graph, (String)description, false, monitor);
            this.setGraphLocation(display, displayListener);
        }
        catch (GraphException e) {
            Msg.showError((Object)((Object)this), null, (String)"Graph Error", (Object)e.getMessage());
        }
        catch (CancelledException e1) {
            return;
        }
    }

    private void setGraphLocation(GraphDisplay display, ASTGraphDisplayListener displayListener) {
        if (this.location == null) {
            return;
        }
        AttributedVertex vertex = displayListener.getVertex(this.location);
        if (vertex == null) {
            return;
        }
        display.setFocusedVertex(vertex, EventTrigger.INTERNAL_ONLY);
    }

    protected void createDataFlowGraph(AttributedGraph graph, TaskMonitor monitor) throws CancelledException {
        Iterator opIter = this.hfunction.getPcodeOps();
        while (opIter.hasNext()) {
            monitor.checkCanceled();
            this.graphOpData(graph, (PcodeOpAST)opIter.next(), monitor);
        }
    }

    private void graphOpData(AttributedGraph graph, PcodeOpAST op, TaskMonitor monitor) throws CancelledException {
        if (op == null || op.getOpcode() == 61) {
            return;
        }
        AttributedVertex opVertex = this.getOpVertex(graph, op, monitor);
        Varnode output = op.getOutput();
        if (output != null) {
            opVertex = this.getOpVertex(graph, op, monitor);
            AttributedVertex outVertex = this.getDataVertex(graph, output, monitor);
            graph.addEdge(opVertex, outVertex);
        }
        int start = 0;
        int stop = op.getNumInputs() - 1;
        switch (op.getOpcode()) {
            case 2: 
            case 3: 
            case 4: 
            case 7: {
                start = 1;
                break;
            }
            case 61: {
                stop = 1;
            }
        }
        for (int i = start; i <= stop; ++i) {
            monitor.checkCanceled();
            Varnode input = op.getInput(i);
            if (input == null) continue;
            if (opVertex == null) {
                opVertex = this.getOpVertex(graph, op, monitor);
            }
            AttributedVertex inVertex = this.getDataVertex(graph, input, monitor);
            graph.addEdge(inVertex, opVertex);
        }
    }

    private AttributedVertex getOpVertex(AttributedGraph graph, PcodeOpAST op, TaskMonitor monitor) {
        String key = "O_" + Integer.toString(op.getSeqnum().getTime());
        AttributedVertex vertex = graph.getVertex(key);
        if (vertex == null) {
            vertex = graph.addVertex(key, key);
            this.setOpVertexAttributes(vertex, op);
        }
        return vertex;
    }

    private void setOpVertexAttributes(AttributedVertex vertex, PcodeOpAST op) {
        vertex.setAttribute(CODE_ATTRIBUTE, this.formatOpMnemonic((PcodeOp)op));
        String vertexType = BODY_NODE;
        switch (op.getOpcode()) {
            case 4: 
            case 5: 
            case 6: 
            case 7: 
            case 8: {
                vertexType = SWITCH_NODE;
                break;
            }
            case 10: {
                vertexType = EXIT_NODE;
            }
        }
        vertex.setAttribute(VERTEX_TYPE_ATTRIBUTE, vertexType);
    }

    private AttributedVertex getDataVertex(AttributedGraph graph, Varnode node, TaskMonitor monitor) {
        Object key;
        AttributedVertex vertex = null;
        HighVariable var = node.getHigh();
        if (var != null) {
            key = "V_" + var.getName();
            vertex = graph.getVertex((String)key);
        } else {
            key = Integer.toString(++this.uniqueNum);
        }
        if (vertex == null) {
            vertex = graph.addVertex((String)key, (String)key);
            this.setVarnodeVertexAttributes(vertex, node);
        }
        return vertex;
    }

    private void setVarnodeVertexAttributes(AttributedVertex vertex, Varnode node) {
        Object label = "";
        HighVariable var = node.getHigh();
        if (var != null) {
            label = var.getName() + ": ";
        }
        label = (String)label + this.translateVarnode(node, false);
        vertex.setAttribute(CODE_ATTRIBUTE, (String)label);
        vertex.setAttribute(VERTEX_TYPE_ATTRIBUTE, DATA_NODE);
    }

    protected void createControlFlowGraph(AttributedGraph graph, TaskMonitor monitor) throws CancelledException {
        Iterator pblockIter = this.hfunction.getBasicBlocks().iterator();
        while (pblockIter.hasNext()) {
            monitor.checkCanceled();
            this.graphPcodeBlock(graph, (PcodeBlock)pblockIter.next(), monitor);
        }
    }

    private void graphPcodeBlock(AttributedGraph graph, PcodeBlock pblock, TaskMonitor monitor) throws CancelledException {
        if (pblock == null) {
            return;
        }
        AttributedVertex fromVertex = this.getBlockVertex(graph, pblock, monitor);
        int outCnt = pblock.getOutSize();
        for (int i = 0; i < outCnt; ++i) {
            monitor.checkCanceled();
            PcodeBlock outPBlock = pblock.getOut(i);
            AttributedVertex toVertex = this.getBlockVertex(graph, outPBlock, monitor);
            graph.addEdge(fromVertex, toVertex);
        }
    }

    private AttributedVertex getBlockVertex(AttributedGraph graph, PcodeBlock pblock, TaskMonitor monitor) {
        String key = Integer.toString(pblock.getIndex());
        AttributedVertex vertex = graph.getVertex(key);
        if (vertex == null) {
            vertex = graph.addVertex(key, key);
            if (pblock instanceof PcodeBlockBasic) {
                this.setBlockVertexAttributes(vertex, (PcodeBlockBasic)pblock);
            } else {
                vertex.setAttribute(CODE_ATTRIBUTE, "<???>");
                vertex.setAttribute(VERTEX_TYPE_ATTRIBUTE, BAD_NODE);
            }
        }
        return vertex;
    }

    private void setBlockVertexAttributes(AttributedVertex vertex, PcodeBlockBasic basicBlk) {
        StringBuffer buf = new StringBuffer();
        int cnt = 0;
        Iterator opIter = basicBlk.getIterator();
        while (opIter.hasNext()) {
            PcodeOp op = (PcodeOp)opIter.next();
            if (buf.length() != 0) {
                buf.append('\n');
            }
            this.formatOp(op, buf);
            if (++cnt != this.codeLimitPerBlock) continue;
            buf.append("\n...");
            break;
        }
        vertex.setAttribute(CODE_ATTRIBUTE, buf.toString());
        String vertexType = BODY_NODE;
        if (basicBlk.getInSize() == 0) {
            vertexType = ENTRY_NODE;
        } else {
            switch (basicBlk.getOutSize()) {
                case 0: {
                    vertexType = EXIT_NODE;
                    break;
                }
                case 1: {
                    vertexType = BODY_NODE;
                    break;
                }
                default: {
                    vertexType = SWITCH_NODE;
                }
            }
        }
        vertex.setAttribute(VERTEX_TYPE_ATTRIBUTE, vertexType);
    }

    private String formatOpMnemonic(PcodeOp op) {
        Object str = op.getMnemonic();
        Varnode output = op.getOutput();
        String size = null;
        if (output != null) {
            switch (output.getSize()) {
                case 1: {
                    size = "b";
                    break;
                }
                case 2: {
                    size = "w";
                    break;
                }
                case 4: {
                    size = "d";
                    break;
                }
                case 8: {
                    size = "q";
                }
            }
            if (size != null) {
                str = (String)str + "." + size;
            }
        }
        return str;
    }

    private void formatOp(PcodeOp op, StringBuffer buf) {
        Varnode output = op.getOutput();
        if (output != null) {
            buf.append(this.translateVarnode(output, true));
            buf.append(" = ");
        }
        buf.append(this.formatOpMnemonic(op));
        buf.append(" ");
        Varnode[] inputs = op.getInputs();
        for (int i = 0; i < inputs.length; ++i) {
            if (i != 0) {
                buf.append(",");
            }
            buf.append(this.translateVarnode(inputs[i], true));
        }
    }

    private String translateVarnode(Varnode node, boolean useVarName) {
        if (node == null) {
            return "null";
        }
        Program p = this.hfunction.getFunction().getProgram();
        Address addr = node.getAddress();
        if (node.isConstant()) {
            return "#" + NumericUtilities.toHexString((long)addr.getOffset(), (int)node.getSize());
        }
        if (node.isUnique()) {
            return "u_" + Long.toHexString(addr.getOffset());
        }
        if (addr.isRegisterAddress()) {
            Register r = p.getRegister(addr, node.getSize());
            if (r == null) {
                r = p.getRegister(addr);
            }
            if (r != null) {
                return r.getName();
            }
        } else {
            if (addr.isStackAddress()) {
                HighVariable var;
                if (useVarName && (var = node.getHigh()) != null) {
                    return var.getName();
                }
                return "Stack[" + NumericUtilities.toSignedHexString((long)addr.getOffset()) + "]";
            }
            if (addr.isMemoryAddress()) {
                return addr.toString(true);
            }
        }
        return node.toString();
    }

    static enum GraphType {
        CONTROL_FLOW_GRAPH("AST Control Flow"),
        DATA_FLOW_GRAPH("AST Data Flow");

        private String name;

        private GraphType(String name) {
            this.name = name;
        }

        public String getName() {
            return this.name;
        }
    }
}

