/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.util.bin.format.elf;

import ghidra.app.cmd.refs.RemoveReferenceCmd;
import ghidra.app.util.bin.format.elf.ElfDynamicTable;
import ghidra.app.util.bin.format.elf.ElfDynamicType;
import ghidra.app.util.bin.format.elf.ElfHeader;
import ghidra.app.util.bin.format.elf.ElfLoadHelper;
import ghidra.app.util.bin.format.elf.ElfProgramHeader;
import ghidra.app.util.bin.format.elf.ElfRelocation;
import ghidra.app.util.bin.format.elf.ElfRelocationTable;
import ghidra.framework.model.DomainObject;
import ghidra.framework.store.LockException;
import ghidra.program.disassemble.Disassembler;
import ghidra.program.disassemble.DisassemblerMessageListener;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressOutOfBoundsException;
import ghidra.program.model.address.AddressOverflowException;
import ghidra.program.model.address.AddressSet;
import ghidra.program.model.address.AddressSetView;
import ghidra.program.model.address.AddressSpace;
import ghidra.program.model.data.DataType;
import ghidra.program.model.data.DataTypeManager;
import ghidra.program.model.data.Pointer;
import ghidra.program.model.data.Pointer32DataType;
import ghidra.program.model.data.Pointer64DataType;
import ghidra.program.model.data.PointerDataType;
import ghidra.program.model.listing.Data;
import ghidra.program.model.listing.Listing;
import ghidra.program.model.listing.Program;
import ghidra.program.model.mem.Memory;
import ghidra.program.model.mem.MemoryBlock;
import ghidra.program.model.symbol.Reference;
import ghidra.program.model.symbol.SourceType;
import ghidra.program.model.symbol.Symbol;
import ghidra.program.model.symbol.SymbolTable;
import ghidra.program.model.util.CodeUnitInsertionException;
import ghidra.util.Msg;
import ghidra.util.exception.AssertException;
import ghidra.util.exception.CancelledException;
import ghidra.util.exception.NotFoundException;
import ghidra.util.task.TaskMonitor;
import java.util.Iterator;

public class ElfDefaultGotPltMarkup {
    private ElfLoadHelper elfLoadHelper;
    private ElfHeader elf;
    private Program program;
    private Listing listing;
    private Memory memory;

    public ElfDefaultGotPltMarkup(ElfLoadHelper elfLoadHelper) {
        this.elfLoadHelper = elfLoadHelper;
        this.elf = elfLoadHelper.getElfHeader();
        this.program = elfLoadHelper.getProgram();
        this.listing = this.program.getListing();
        this.memory = this.program.getMemory();
    }

    private void log(String msg) {
        this.elfLoadHelper.log(msg);
    }

    public void process(TaskMonitor monitor) throws CancelledException {
        if (this.elf.e_shnum() == 0) {
            this.processDynamicPLTGOT(monitor);
        } else {
            this.processGOTSections(monitor);
            this.processPLTSection(monitor);
        }
    }

    private void processGOTSections(TaskMonitor monitor) throws CancelledException {
        MemoryBlock[] blocks;
        for (MemoryBlock gotBlock : blocks = this.memory.getBlocks()) {
            monitor.checkCanceled();
            if (!gotBlock.getName().startsWith(".got")) continue;
            gotBlock.setWrite(false);
            this.processGOT(gotBlock.getStart(), gotBlock.getEnd(), monitor);
        }
    }

    private void processDynamicPLTGOT(TaskMonitor monitor) throws CancelledException {
        ElfDynamicTable dynamicTable = this.elf.getDynamicTable();
        if (dynamicTable == null || !dynamicTable.containsDynamicValue(ElfDynamicType.DT_PLTGOT) || !dynamicTable.containsDynamicValue(ElfDynamicType.DT_JMPREL)) {
            return;
        }
        AddressSpace defaultSpace = this.program.getAddressFactory().getDefaultAddressSpace();
        long imageBaseAdj = this.elfLoadHelper.getImageBaseWordAdjustmentOffset();
        try {
            long relocTableAddr = this.elf.adjustAddressForPrelink(dynamicTable.getDynamicValue(ElfDynamicType.DT_JMPREL));
            ElfProgramHeader relocTableLoadHeader = this.elf.getProgramLoadHeaderContaining(relocTableAddr);
            if (relocTableLoadHeader == null || relocTableLoadHeader.getOffset() < 0L) {
                return;
            }
            long relocTableOffset = relocTableLoadHeader.getOffset(relocTableAddr);
            ElfRelocationTable relocationTable = this.elf.getRelocationTableAtOffset(relocTableOffset);
            if (relocationTable == null) {
                return;
            }
            long pltgot = this.elf.adjustAddressForPrelink(dynamicTable.getDynamicValue(ElfDynamicType.DT_PLTGOT));
            ElfRelocation[] relocations = relocationTable.getRelocations();
            long lastGotOffset = relocations[relocations.length - 1].getOffset();
            if (lastGotOffset < pltgot) {
                return;
            }
            Address gotStart = defaultSpace.getAddress(pltgot + imageBaseAdj);
            Address gotEnd = defaultSpace.getAddress(lastGotOffset + imageBaseAdj);
            this.processGOT(gotStart, gotEnd, monitor);
            this.processDynamicPLT(gotStart, gotEnd, monitor);
        }
        catch (NotFoundException e) {
            throw new AssertException((Throwable)e);
        }
        catch (AddressOutOfBoundsException e) {
            this.log("Failed to process GOT: " + e.getMessage());
        }
    }

    private void processGOT(Address gotStart, Address gotEnd, TaskMonitor monitor) throws CancelledException {
        boolean imageBaseAlreadySet = this.elf.isPreLinked();
        try {
            Address newImageBase = null;
            while (gotStart.compareTo((Object)gotEnd) <= 0) {
                monitor.checkCanceled();
                Data data = this.createPointer(gotStart, true);
                try {
                    gotStart = data.getMaxAddress().add(1L);
                }
                catch (AddressOutOfBoundsException e) {
                    break;
                }
                newImageBase = this.UglyImageBaseCheck(data, newImageBase);
            }
            if (newImageBase != null) {
                this.log("Invalid Address found in .got table.  Suspect Prelinked shared object file");
                if (imageBaseAlreadySet) {
                    this.log("ERROR: Unable to adjust image base for pre-link - retaining existing image base of " + this.program.getImageBase());
                } else {
                    this.program.setImageBase(newImageBase, true);
                    this.log("Setting Image base to: " + newImageBase);
                    imageBaseAlreadySet = true;
                }
            }
        }
        catch (CodeUnitInsertionException e) {
            this.log("Failed to process GOT: " + e.getMessage());
        }
        catch (AddressOverflowException e) {
            this.log("Failed to adjust image base: " + e.getMessage());
        }
        catch (LockException e) {
            throw new AssertException((Throwable)e);
        }
    }

    private void processPLTSection(TaskMonitor monitor) throws CancelledException {
        if (this.elf.isRelocatable()) {
            return;
        }
        MemoryBlock pltBlock = this.memory.getBlock(".plt");
        if (pltBlock == null || !pltBlock.isExecute() || pltBlock.getSize() <= 16L) {
            return;
        }
        int skipPointers = 16;
        if (this.elf.e_machine() == 40 || this.elf.e_machine() == 183) {
            skipPointers = 0;
        }
        Address minAddress = pltBlock.getStart().add((long)skipPointers);
        Address maxAddress = pltBlock.getEnd();
        this.processLinkageTable(".plt", minAddress, maxAddress, monitor);
    }

    private void processDynamicPLT(Address gotStart, Address gotEnd, TaskMonitor monitor) throws CancelledException {
        Address pltStart = null;
        Address pltEnd = null;
        for (Data gotPtr : this.listing.getDefinedData((AddressSetView)new AddressSet(gotStart.next(), gotEnd), true)) {
            MemoryBlock block;
            monitor.checkCanceled();
            if (!gotPtr.isPointer()) {
                Msg.error((Object)this, (Object)"ELF PLTGOT contains non-pointer");
                return;
            }
            Address ptr = (Address)gotPtr.getValue();
            if (ptr.getOffset() == 0L || (block = this.memory.getBlock(ptr)) == null || block.getName().equals("EXTERNAL")) continue;
            if (pltStart == null) {
                pltStart = ptr;
            }
            pltEnd = ptr;
        }
        if (pltStart != null) {
            this.processLinkageTable("PLT", pltStart, pltEnd, monitor);
        }
    }

    public void processLinkageTable(String pltName, Address minAddress, Address maxAddress, TaskMonitor monitor) throws CancelledException {
        this.disassemble(minAddress, maxAddress, this.program, monitor);
        int count = this.convertSymbolsToExternalFunctions(minAddress, maxAddress);
        if (count > 0) {
            this.log("Converted " + count + " " + pltName + " section symbols to external thunks");
        }
    }

    private int convertSymbolsToExternalFunctions(Address minAddress, Address maxAddress) {
        Symbol s;
        Address symAddr;
        AddressSet set = new AddressSet();
        SymbolTable symbolTable = this.program.getSymbolTable();
        Iterator iterator = symbolTable.getPrimarySymbolIterator(minAddress, true).iterator();
        while (iterator.hasNext() && (symAddr = (s = (Symbol)iterator.next()).getAddress()).compareTo((Object)maxAddress) <= 0) {
            if (s.getSource() == SourceType.DEFAULT || this.listing.getDataAt(symAddr) != null) continue;
            set.add(symAddr);
        }
        if (set.isEmpty()) {
            return 0;
        }
        for (Address addr : set.getAddresses(true)) {
            Symbol s2 = symbolTable.getPrimarySymbol(addr);
            this.elfLoadHelper.createExternalFunctionLinkage(s2.getName(), addr, null);
        }
        return (int)set.getNumAddresses();
    }

    private void disassemble(Address start, Address end, Program prog, TaskMonitor monitor) throws CancelledException {
        DisassemblerMessageListener dml = msg -> {};
        AddressSet set = new AddressSet(start, end);
        Disassembler disassembler = Disassembler.getDisassembler((Program)prog, (TaskMonitor)monitor, (DisassemblerMessageListener)dml);
        while (!set.isEmpty()) {
            monitor.checkCanceled();
            AddressSet disset = disassembler.disassemble(set.getMinAddress(), (AddressSetView)set, true);
            if (disset.isEmpty()) {
                prog.getBookmarkManager().removeBookmarks((AddressSetView)set, "Error", "Bad Instruction", monitor);
                break;
            }
            set.delete((AddressSetView)disset);
        }
    }

    private Data createPointer(Address addr, boolean keepRefWhenValid) throws CodeUnitInsertionException {
        MemoryBlock block = this.memory.getBlock(addr);
        if (block == null || !block.isInitialized()) {
            return null;
        }
        int pointerSize = this.program.getDataTypeManager().getDataOrganization().getPointerSize();
        Pointer pointer = PointerDataType.dataType.clone((DataTypeManager)this.program.getDataTypeManager());
        if (this.elf.is32Bit() && pointerSize != 4) {
            pointer = Pointer32DataType.dataType;
        } else if (this.elf.is64Bit() && pointerSize != 8) {
            pointer = Pointer64DataType.dataType;
        }
        Data data = this.listing.getDataAt(addr);
        if (data == null || !pointer.isEquivalent(data.getDataType())) {
            if (data != null) {
                this.listing.clearCodeUnits(addr, addr.add((long)(pointerSize - 1)), false);
            }
            data = this.listing.createData(addr, (DataType)pointer);
        }
        if (keepRefWhenValid && ElfDefaultGotPltMarkup.isValidPointer(data)) {
            ElfDefaultGotPltMarkup.setConstant(data);
        } else {
            this.removeMemRefs(data);
        }
        return data;
    }

    public static void setConstant(Data data) {
        Memory memory = data.getProgram().getMemory();
        MemoryBlock block = memory.getBlock(data.getAddress());
        if (!block.isWrite() || block.getName().startsWith(".got")) {
            return;
        }
        data.setLong("mutability", 2L);
    }

    public static boolean isValidPointer(Data pointerData) {
        Address refAddr;
        Program program = pointerData.getProgram();
        Memory memory = program.getMemory();
        if (memory.contains(refAddr = (Address)pointerData.getValue())) {
            return true;
        }
        Symbol[] syms = program.getSymbolTable().getSymbols(refAddr);
        return syms != null && syms.length > 0 && syms[0].getSource() != SourceType.DEFAULT;
    }

    private void removeMemRefs(Data data) {
        if (data != null) {
            Reference[] refs;
            for (Reference ref : refs = data.getValueReferences()) {
                RemoveReferenceCmd cmd = new RemoveReferenceCmd(ref);
                cmd.applyTo((DomainObject)data.getProgram());
            }
        }
    }

    private Address UglyImageBaseCheck(Data data, Address imageBase) {
        if (this.elf.e_machine() != 40) {
            return null;
        }
        if (!this.elf.isSharedObject()) {
            return null;
        }
        if (imageBase != null) {
            return imageBase;
        }
        Object dValue = data.getValue();
        if (dValue == null || !(dValue instanceof Address)) {
            return null;
        }
        Address daddr = (Address)dValue;
        if (this.memory.contains(daddr)) {
            return null;
        }
        if (daddr.getOffset() < 4L) {
            return null;
        }
        if (this.program.getImageBase().getOffset() != 0L) {
            return null;
        }
        if (this.program.getRelocationTable().getRelocation(data.getAddress()) != null) {
            return null;
        }
        MemoryBlock tBlock = this.memory.getBlock(".text");
        if (tBlock == null) {
            return null;
        }
        Address topAddr = tBlock.getEnd();
        long byteMask = -1L;
        for (long topVal = topAddr.getOffset(); topVal != 0L; topVal >>>= 8) {
            byteMask <<= 8;
        }
        long newBase = daddr.getOffset() & byteMask;
        if (newBase == 0L) {
            return null;
        }
        return daddr.getNewAddress(newBase);
    }
}

