/*
 * Decompiled with CFR 0.152.
 */
package ghidra.file.formats.ext4;

import ghidra.app.util.bin.BinaryReader;
import ghidra.app.util.bin.ByteProvider;
import ghidra.file.formats.ext4.Ext4DirEntry;
import ghidra.file.formats.ext4.Ext4DirEntry2;
import ghidra.file.formats.ext4.Ext4Extent;
import ghidra.file.formats.ext4.Ext4ExtentHeader;
import ghidra.file.formats.ext4.Ext4ExtentIdx;
import ghidra.file.formats.ext4.Ext4File;
import ghidra.file.formats.ext4.Ext4FileSystemFactory;
import ghidra.file.formats.ext4.Ext4GroupDescriptor;
import ghidra.file.formats.ext4.Ext4IBlock;
import ghidra.file.formats.ext4.Ext4Inode;
import ghidra.file.formats.ext4.Ext4SuperBlock;
import ghidra.formats.gfilesystem.FSRLRoot;
import ghidra.formats.gfilesystem.FSUtilities;
import ghidra.formats.gfilesystem.FileSystemIndexHelper;
import ghidra.formats.gfilesystem.FileSystemRefManager;
import ghidra.formats.gfilesystem.GFile;
import ghidra.formats.gfilesystem.GFileSystem;
import ghidra.formats.gfilesystem.annotations.FileSystemInfo;
import ghidra.util.NumericUtilities;
import ghidra.util.exception.CancelledException;
import ghidra.util.task.TaskMonitor;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.List;

@FileSystemInfo(type="ext4", description="EXT4", factory=Ext4FileSystemFactory.class)
public class Ext4FileSystem
implements GFileSystem {
    public static final Charset EXT4_DEFAULT_CHARSET = StandardCharsets.UTF_8;
    private FileSystemIndexHelper<Ext4File> fsih;
    private FileSystemRefManager refManager = new FileSystemRefManager((GFileSystem)this);
    private FSRLRoot fsrl;
    private int blockSize;
    private ByteProvider provider;
    private String volumeName;
    private String uuid;
    private static final int MAX_SYMLINK_LOOKUP_COUNT = 100;

    public Ext4FileSystem(FSRLRoot fsrl, ByteProvider provider) {
        this.fsrl = fsrl;
        this.fsih = new FileSystemIndexHelper((GFileSystem)this, fsrl);
        this.provider = provider;
    }

    public void mountFS(TaskMonitor monitor) throws IOException, CancelledException {
        BinaryReader reader = new BinaryReader(this.provider, true);
        reader.setPointerIndex(1024);
        Ext4SuperBlock superBlock = new Ext4SuperBlock(reader);
        this.volumeName = superBlock.getVolumeName();
        this.uuid = NumericUtilities.convertBytesToString((byte[])superBlock.getS_uuid());
        int s_log_block_size = superBlock.getS_log_block_size();
        this.blockSize = (int)Math.pow(2.0, 10 + s_log_block_size);
        int groupSize = this.blockSize * superBlock.getS_blocks_per_group();
        if (groupSize <= 0) {
            throw new IOException("Invalid groupSize: " + groupSize);
        }
        int numGroups = (int)(this.provider.length() / (long)groupSize);
        if (this.provider.length() % (long)groupSize != 0L) {
            ++numGroups;
        }
        boolean is64Bit = superBlock.getS_desc_size() > 32 && (superBlock.getS_feature_incompat() & 0x80) > 0;
        int groupDescriptorOffset = this.blockSize;
        reader.setPointerIndex(groupDescriptorOffset);
        Ext4GroupDescriptor[] groupDescriptors = new Ext4GroupDescriptor[numGroups];
        for (int i = 0; i < numGroups; ++i) {
            monitor.checkCanceled();
            groupDescriptors[i] = new Ext4GroupDescriptor(reader, is64Bit);
            monitor.incrementProgress(1L);
        }
        Ext4Inode[] inodes = this.getInodes(reader, superBlock, groupDescriptors, is64Bit, monitor);
        int s_inodes_count = superBlock.getS_inodes_count();
        for (int i = 0; i < s_inodes_count; ++i) {
            Ext4Inode inode = inodes[i];
            if (inode == null) continue;
            if ((inode.getI_mode() & 0xF000) == 16384) {
                this.processDirectory(reader, superBlock, inodes, i, null, null, monitor);
                continue;
            }
            if ((inode.getI_mode() & 0xF000) != 32768) continue;
            this.processFile(reader, superBlock, inode, monitor);
        }
    }

    private void processDirectory(BinaryReader reader, Ext4SuperBlock superBlock, Ext4Inode[] inodes, int index, String name, GFile parent, TaskMonitor monitor) throws IOException, CancelledException {
        boolean isDirEntry2;
        if (name != null && (name.equals(".") || name.equals(".."))) {
            return;
        }
        Ext4Inode inode = inodes[index];
        if (name == null && parent == null) {
            parent = this.fsih.getRootDir();
        } else {
            if (parent == null) {
                parent = this.fsih.getRootDir();
            }
            parent = this.fsih.storeFileWithParent(name, parent, -1, true, (long)(inode.getI_size_high() << 32 | inode.getI_size_lo()), (Object)new Ext4File(name, inode));
        }
        if ((inode.getI_flags() & 0x80000) == 0) {
            return;
        }
        boolean bl = isDirEntry2 = (superBlock.getS_feature_incompat() & 2) != 0;
        if ((inode.getI_flags() & 0x80000) == 0) {
            throw new IOException("File system fails to use extents.");
        }
        Ext4IBlock i_block = inode.getI_block();
        this.processIBlock(reader, superBlock, inodes, parent, isDirEntry2, i_block, monitor);
        inodes[index] = null;
    }

    private void processIBlock(BinaryReader reader, Ext4SuperBlock superBlock, Ext4Inode[] inodes, GFile parent, boolean isDirEntry2, Ext4IBlock i_block, TaskMonitor monitor) throws CancelledException, IOException {
        Ext4ExtentHeader header = i_block.getHeader();
        if (header.getEh_depth() == 0) {
            int numEntries = header.getEh_entries();
            List<Ext4Extent> entries = i_block.getExtentEntries();
            for (int i = 0; i < numEntries; ++i) {
                monitor.checkCanceled();
                Ext4Extent extent = entries.get(i);
                long low = (long)extent.getEe_start_lo() & 0xFFFFFFFFL;
                long high = (long)extent.getEe_start_hi() & 0xFFFFFFFFL;
                long blockNumber = high << 16 | low;
                long offset = blockNumber * (long)this.blockSize;
                reader.setPointerIndex(offset);
                if (isDirEntry2) {
                    this.processDirEntry2(reader, superBlock, inodes, parent, monitor, extent, offset);
                    continue;
                }
                this.processDirEntry(reader, superBlock, inodes, parent, monitor, extent, offset);
            }
        } else {
            int numEntries = header.getEh_entries();
            List<Ext4ExtentIdx> entries = i_block.getIndexEntries();
            for (int i = 0; i < numEntries; ++i) {
                monitor.checkCanceled();
                Ext4ExtentIdx extentIndex = entries.get(i);
                long lo = extentIndex.getEi_leaf_lo();
                long hi = extentIndex.getEi_leaf_hi();
                long physicalBlockOfNextLevel = hi << 16 | lo;
                long offset = physicalBlockOfNextLevel * (long)this.blockSize;
                reader.setPointerIndex(offset);
                Ext4IBlock intermediateBlock = new Ext4IBlock(reader, true);
                this.processIBlock(reader, superBlock, inodes, parent, isDirEntry2, intermediateBlock, monitor);
            }
        }
    }

    private void processDirEntry(BinaryReader reader, Ext4SuperBlock superBlock, Ext4Inode[] inodes, GFile parent, TaskMonitor monitor, Ext4Extent extent, long offset) throws CancelledException, IOException {
        while (reader.getPointerIndex() - offset < (long)extent.getEe_len() * (long)this.blockSize) {
            monitor.checkCanceled();
            if (reader.peekNextInt() == 0) {
                return;
            }
            Ext4DirEntry dirEnt = new Ext4DirEntry(reader);
            int childIndex = dirEnt.getInode();
            Ext4Inode child = inodes[childIndex];
            if ((child.getI_mode() & 0xF000) == 16384) {
                String childName = dirEnt.getName();
                long readerOffset = reader.getPointerIndex();
                this.processDirectory(reader, superBlock, inodes, childIndex, childName, parent, monitor);
                reader.setPointerIndex(readerOffset);
                continue;
            }
            if ((child.getI_mode() & 0xF000) == 32768 || (child.getI_mode() & 0xF000) == 40960) {
                this.storeFile(inodes, dirEnt, parent);
                continue;
            }
            throw new IOException("Inode " + dirEnt.getInode() + " has unhandled file type: " + (child.getI_mode() & 0xF000));
        }
    }

    private void processDirEntry2(BinaryReader reader, Ext4SuperBlock superBlock, Ext4Inode[] inodes, GFile parent, TaskMonitor monitor, Ext4Extent extent, long offset) throws CancelledException, IOException {
        while (reader.getPointerIndex() - offset < (long)extent.getEe_len() * (long)this.blockSize) {
            monitor.checkCanceled();
            if (reader.peekNextInt() == 0) {
                return;
            }
            Ext4DirEntry2 dirEnt2 = new Ext4DirEntry2(reader);
            if (dirEnt2.getFile_type() == 2) {
                int childInode = dirEnt2.getInode();
                String childName = dirEnt2.getName();
                long readerOffset = reader.getPointerIndex();
                this.processDirectory(reader, superBlock, inodes, childInode, childName, parent, monitor);
                reader.setPointerIndex(readerOffset);
                continue;
            }
            if (dirEnt2.getFile_type() == 1 || dirEnt2.getFile_type() == 7) {
                this.storeFile(inodes, dirEnt2, parent);
                continue;
            }
            throw new IOException("Inode " + dirEnt2.getInode() + " has unhandled file type: " + dirEnt2.getFile_type());
        }
    }

    private void storeFile(Ext4Inode[] inodes, Ext4DirEntry dirEnt, GFile parent) {
        int fileInodeNum = dirEnt.getInode();
        Ext4Inode fileInode = inodes[fileInodeNum];
        long fileSize = fileInode.getI_size_high() << 32 | fileInode.getI_size_lo();
        this.fsih.storeFileWithParent(dirEnt.getName(), parent, -1, (fileInode.getI_mode() & 0xF000) == 16384, fileSize, (Object)new Ext4File(dirEnt.getName(), fileInode));
        inodes[fileInodeNum] = null;
    }

    private void storeFile(Ext4Inode[] inodes, Ext4DirEntry2 dirEnt2, GFile parent) {
        int fileInodeNum = dirEnt2.getInode();
        Ext4Inode fileInode = inodes[fileInodeNum];
        if (fileInode == null) {
            return;
        }
        long fileSize = fileInode.getI_size_high() << 32 | fileInode.getI_size_lo();
        this.fsih.storeFileWithParent(dirEnt2.getName(), parent, -1, dirEnt2.getFile_type() == 2, fileSize, (Object)new Ext4File(dirEnt2.getName(), fileInode));
        inodes[fileInodeNum] = null;
    }

    private void processFile(BinaryReader reader, Ext4SuperBlock superBlock, Ext4Inode inode, TaskMonitor monitor) {
    }

    public int getFileCount() {
        return this.fsih.getFileCount();
    }

    public List<GFile> getListing(GFile directory) throws IOException {
        return this.fsih.getListing(directory);
    }

    public String getInfo(GFile file, TaskMonitor monitor) {
        Ext4File ext4File = (Ext4File)this.fsih.getMetadata(file);
        if (ext4File == null) {
            return null;
        }
        Ext4Inode inode = ext4File.getInode();
        Object info = "";
        long size = inode.getI_size_high() << 32 | inode.getI_size_lo();
        if ((inode.getI_mode() & 0xF000) == 40960) {
            Ext4IBlock block = inode.getI_block();
            byte[] extra = block.getExtra();
            info = "Symlink to \"" + new String(extra).trim() + "\"\n";
        } else {
            info = "File size:  0x" + Long.toHexString(size);
        }
        return info;
    }

    public InputStream getInputStream(GFile file, TaskMonitor monitor) throws IOException, CancelledException {
        Ext4File extFile = (Ext4File)this.fsih.getMetadata(file);
        if (extFile == null) {
            return null;
        }
        Ext4Inode inode = extFile.getInode();
        if (inode == null) {
            return null;
        }
        if ((inode.getI_mode() & 0xF000) == 16384) {
            throw new IOException(extFile.getName() + " is a directory.");
        }
        if ((inode.getI_mode() & 0xF000) == 40960 && (inode = this.resolveSymLink(file)) == null) {
            throw new IOException(extFile.getName() + " is a broken symlink.");
        }
        return this.getInputStream(inode);
    }

    private Ext4Inode resolveSymLink(GFile file) throws IOException {
        for (int lookupCount = 0; file != null && lookupCount < 100; ++lookupCount) {
            Ext4File extFile = (Ext4File)this.fsih.getMetadata(file);
            Ext4Inode inode = extFile.getInode();
            if ((inode.getI_mode() & 0xF000) != 40960) {
                return inode;
            }
            Ext4IBlock block = inode.getI_block();
            byte[] extra = block.getExtra();
            String symlinkDestPath = new String(extra).trim();
            if (!symlinkDestPath.startsWith("/")) {
                if (file.getParentFile() == null) {
                    throw new IOException("Not parent file for " + file);
                }
                symlinkDestPath = FSUtilities.appendPath((String[])new String[]{file.getParentFile().getPath(), symlinkDestPath});
            }
            file = this.lookup(symlinkDestPath);
        }
        return null;
    }

    private InputStream getInputStream(Ext4Inode inode) throws IOException {
        Ext4IBlock i_block;
        Ext4ExtentHeader header;
        boolean usesExtents;
        int i_size_lo = inode.getI_size_lo();
        int i_size_high = inode.getI_size_high();
        long size = i_size_high << 32 | i_size_lo;
        boolean bl = usesExtents = (inode.getI_flags() & 0x80000) != 0;
        if (usesExtents && (header = (i_block = inode.getI_block()).getHeader()).getEh_depth() == 0) {
            List<Ext4Extent> entries = i_block.getExtentEntries();
            return this.concatenateExtents(entries, size);
        }
        return null;
    }

    private InputStream concatenateExtents(List<Ext4Extent> entries, long actualSize) throws IOException {
        if (actualSize > Integer.MAX_VALUE) {
            throw new IOException("File is >2GB, too large to extract.  Please report to Ghidra team.");
        }
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        for (int i = 0; i < entries.size(); ++i) {
            Ext4Extent extent = entries.get(i);
            long low = (long)extent.getEe_start_lo() & 0xFFFFFFFFL;
            long high = (long)extent.getEe_start_hi() & 0xFFFFFFFFL;
            long blockNumber = high << 16 | low;
            long extentOffset = blockNumber * (long)this.blockSize;
            long extentSize = ((long)extent.getEe_len() & 0xFFFFL) * (long)this.blockSize;
            try {
                byte[] extentBytes = this.provider.readBytes(extentOffset, extentSize);
                baos.write(extentBytes);
                continue;
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        return new ByteArrayInputStream(baos.toByteArray(), 0, (int)actualSize);
    }

    private Ext4Inode[] getInodes(BinaryReader reader, Ext4SuperBlock superBlock, Ext4GroupDescriptor[] groupDescriptors, boolean is64Bit, TaskMonitor monitor) throws IOException, CancelledException {
        int inodeCount = superBlock.getS_inodes_count();
        Ext4Inode[] inodes = new Ext4Inode[inodeCount + 1];
        int inodeIndex = 1;
        for (int i = 0; i < groupDescriptors.length; ++i) {
            monitor.checkCanceled();
            long inodeTableBlockOffset = (long)groupDescriptors[i].getBg_inode_table_lo() & 0xFFFFFFFFL;
            if (is64Bit) {
                inodeTableBlockOffset = (long)(groupDescriptors[i].getBg_inode_table_hi() << 32) | inodeTableBlockOffset;
            }
            long offset = inodeTableBlockOffset * (long)this.blockSize;
            reader.setPointerIndex(offset);
            int inodesPerGroup = superBlock.getS_inodes_per_group();
            monitor.setMessage("Reading inode table " + i + " of " + (groupDescriptors.length - 1) + "...");
            monitor.setMaximum((long)inodesPerGroup);
            monitor.setProgress(0L);
            for (int j = 0; j < inodesPerGroup; ++j) {
                monitor.checkCanceled();
                monitor.incrementProgress(1L);
                Ext4Inode inode = new Ext4Inode(reader);
                reader.setPointerIndex(offset += (long)superBlock.getS_inode_size());
                inodes[inodeIndex++] = inode;
            }
        }
        return inodes;
    }

    public void close() throws IOException {
        this.refManager.onClose();
        this.provider.close();
        this.provider = null;
        this.fsih.clear();
    }

    public String getName() {
        return this.fsrl.getContainer().getName() + " - " + this.volumeName + " - " + this.uuid;
    }

    public FSRLRoot getFSRL() {
        return this.fsrl;
    }

    public boolean isClosed() {
        return this.provider == null;
    }

    public FileSystemRefManager getRefManager() {
        return this.refManager;
    }

    public GFile lookup(String path) throws IOException {
        return this.fsih.lookup(path);
    }
}

