/*
 * Decompiled with CFR 0.152.
 */
package ghidra.framework.client;

import docking.widgets.OptionDialog;
import ghidra.framework.client.ClientUtil;
import ghidra.framework.client.NotConnectedException;
import ghidra.framework.client.RemoteAdapterListener;
import ghidra.framework.client.RepositoryAdapter;
import ghidra.framework.client.ServerConnectTask;
import ghidra.framework.model.ServerInfo;
import ghidra.framework.remote.RepositoryHandle;
import ghidra.framework.remote.RepositoryServerHandle;
import ghidra.util.Msg;
import ghidra.util.SystemUtilities;
import ghidra.util.datastruct.WeakDataStructureFactory;
import ghidra.util.datastruct.WeakSet;
import ghidra.util.exception.DuplicateFileException;
import ghidra.util.exception.DuplicateNameException;
import ghidra.util.exception.UserAccessException;
import java.io.EOFException;
import java.io.IOException;
import java.net.ConnectException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.rmi.RemoteException;
import java.security.GeneralSecurityException;
import javax.security.auth.login.LoginException;

public class RepositoryServerAdapter {
    private static final int HOUR = 3600000;
    private final ServerInfo server;
    private final String serverInfoStr;
    private String currentUser = ClientUtil.getUserName();
    private RepositoryServerHandle serverHandle;
    private boolean unexpectedDisconnect = false;
    private boolean connectCancelled = false;
    private WeakSet<RemoteAdapterListener> listenerList = WeakDataStructureFactory.createCopyOnWriteWeakSet();

    RepositoryServerAdapter(ServerInfo server) {
        this.server = server;
        this.serverInfoStr = server.toString();
    }

    protected RepositoryServerAdapter(RepositoryServerHandle serverHandle, String serverInfoString) {
        this.server = null;
        this.serverHandle = serverHandle;
        this.serverInfoStr = serverInfoString;
    }

    public String toString() {
        return this.serverInfoStr;
    }

    public void addListener(RemoteAdapterListener listener) {
        this.listenerList.add((Object)listener);
    }

    public void removeListener(RemoteAdapterListener listener) {
        this.listenerList.remove((Object)listener);
    }

    public boolean isCancelled() {
        return this.connectCancelled;
    }

    private void fireStateChanged() {
        for (RemoteAdapterListener listener : this.listenerList) {
            listener.connectionStateChanged(this);
        }
    }

    public boolean isConnected() {
        return this.serverHandle != null;
    }

    public synchronized boolean connect() throws NotConnectedException {
        this.connectCancelled = false;
        if (this.serverHandle != null && this.verifyConnection()) {
            return true;
        }
        Throwable cause = null;
        try {
            this.serverHandle = ClientUtil.connect(this.server);
            this.unexpectedDisconnect = false;
            if (this.serverHandle != null) {
                Msg.info((Object)this, (Object)("Connected to Ghidra Server at " + this.serverInfoStr));
                this.currentUser = this.serverHandle.getUser();
                this.fireStateChanged();
                this.checkPasswordExpiration();
                return true;
            }
            this.connectCancelled = true;
            return false;
        }
        catch (LoginException e) {
            Msg.showError((Object)this, null, (String)"Server Error", (Object)("Server access denied (" + this.serverInfoStr + ")."));
            cause = e;
        }
        catch (GeneralSecurityException e) {
            Msg.showError((Object)this, null, (String)"Server Error", (Object)("Server access denied (" + this.serverInfoStr + "): " + e.getMessage()));
            cause = e;
        }
        catch (ConnectException | SocketTimeoutException | java.rmi.ConnectException e) {
            Msg.showError((Object)this, null, (String)"Server Error", (Object)("Connection to server failed (" + this.server + ")."));
            cause = e;
        }
        catch (UnknownHostException | java.rmi.UnknownHostException e) {
            Msg.showError((Object)this, null, (String)"Server Error", (Object)("Server Not Found (" + this.server.getServerName() + ")."));
            cause = e;
        }
        catch (RemoteException e) {
            String msg = e.getMessage();
            Throwable t = e;
            while ((t = t.getCause()) != null) {
                String err = t.getMessage();
                msg = err != null ? err : t.toString();
                cause = t;
            }
            Msg.showError((Object)this, null, (String)"Server Error", (Object)("An error occurred on the server (" + this.serverInfoStr + ").\n" + msg), (Throwable)e);
        }
        catch (IOException e) {
            String err = e.getMessage();
            if (err == null && e instanceof EOFException) {
                err = "Ghidra Server process may have died";
            }
            String msg = err != null ? err : e.toString();
            Msg.showError((Object)this, null, (String)"Server Error", (Object)("An error occurred while connecting to the server (" + this.serverInfoStr + ").\n" + msg), (Throwable)e);
        }
        throw new NotConnectedException("Not connected to repository server", cause);
    }

    private void checkPasswordExpiration() {
        try {
            if (!this.serverHandle.canSetPassword()) {
                return;
            }
            long expiration = this.serverHandle.getPasswordExpiration();
            if (expiration >= 0L) {
                Object msg;
                if (expiration == 0L) {
                    msg = "Your server password has expired!\nPlease change immediately.";
                } else {
                    long hours = (expiration + 3600000L - 1L) / 3600000L;
                    msg = "Your password will expire in less than " + hours + " hour(s)!\nPlease change immediately.";
                }
                if (SystemUtilities.isInHeadlessMode()) {
                    Msg.warn((Object)this, (Object)msg);
                } else if (1 == OptionDialog.showOptionDialog(null, (String)"Password Change Required", (String)msg, (String)"OK", (int)2)) {
                    try {
                        ClientUtil.changePassword(null, this.serverHandle, this.serverInfoStr);
                    }
                    catch (IOException e) {
                        Msg.showError(ServerConnectTask.class, null, (String)"Password Change Failed", (Object)"Password changed failed due to server error!", (Throwable)e);
                    }
                }
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    synchronized boolean verifyConnection() {
        if (this.serverHandle == null) {
            return false;
        }
        try {
            this.serverHandle.connected();
        }
        catch (NotConnectedException | RemoteException e) {
            if (!this.recoverConnection(e)) {
                return false;
            }
        }
        catch (IOException e) {
            return false;
        }
        return true;
    }

    private boolean recoverConnection(IOException e) {
        if (this.server == null) {
            return false;
        }
        this.disconnect(true);
        return false;
    }

    public synchronized RepositoryAdapter createRepository(String name) throws DuplicateNameException, UserAccessException, IOException, NotConnectedException {
        this.checkServerHandle();
        try {
            return new RepositoryAdapter(this, name, this.serverHandle.createRepository(name));
        }
        catch (RemoteException e) {
            Throwable t = e.getCause();
            if (t instanceof DuplicateFileException) {
                throw new DuplicateNameException("Repository '" + name + "' already exists");
            }
            if (t instanceof UserAccessException) {
                throw (UserAccessException)t;
            }
            if (this.recoverConnection(e)) {
                return new RepositoryAdapter(this, name, this.serverHandle.createRepository(name));
            }
            throw e;
        }
    }

    public RepositoryAdapter getRepository(String name) {
        return new RepositoryAdapter(this, name);
    }

    synchronized RepositoryHandle getRepositoryHandle(String name) throws IOException, NotConnectedException {
        this.checkServerHandle();
        try {
            return this.serverHandle.getRepository(name);
        }
        catch (RemoteException e) {
            if (this.recoverConnection(e)) {
                return this.serverHandle.getRepository(name);
            }
            throw e;
        }
    }

    public synchronized void deleteRepository(String name) throws UserAccessException, IOException, NotConnectedException {
        this.checkServerHandle();
        try {
            this.serverHandle.deleteRepository(name);
        }
        catch (RemoteException e) {
            if (this.recoverConnection(e)) {
                this.serverHandle.deleteRepository(name);
                return;
            }
            throw e;
        }
    }

    public synchronized String[] getRepositoryNames() throws IOException, NotConnectedException {
        this.checkServerHandle();
        try {
            return this.serverHandle.getRepositoryNames();
        }
        catch (RemoteException e) {
            if (this.recoverConnection(e)) {
                return this.serverHandle.getRepositoryNames();
            }
            throw e;
        }
    }

    public synchronized boolean anonymousAccessAllowed() throws IOException, NotConnectedException {
        this.checkServerHandle();
        try {
            return this.serverHandle.anonymousAccessAllowed();
        }
        catch (RemoteException e) {
            if (this.recoverConnection(e)) {
                return this.serverHandle.anonymousAccessAllowed();
            }
            throw e;
        }
    }

    public synchronized boolean isReadOnly() throws IOException, NotConnectedException {
        this.checkServerHandle();
        try {
            return this.serverHandle.isReadOnly();
        }
        catch (RemoteException e) {
            if (this.recoverConnection(e)) {
                return this.serverHandle.isReadOnly();
            }
            throw e;
        }
    }

    public String getUser() {
        return this.currentUser;
    }

    public synchronized String[] getAllUsers() throws IOException, NotConnectedException {
        this.checkServerHandle();
        try {
            return this.serverHandle.getAllUsers();
        }
        catch (RemoteException e) {
            if (this.recoverConnection(e)) {
                return this.serverHandle.getAllUsers();
            }
            throw e;
        }
    }

    public synchronized boolean setPassword(char[] saltedSHA256PasswordHash) throws IOException, NotConnectedException {
        this.checkServerHandle();
        try {
            return this.serverHandle.setPassword(saltedSHA256PasswordHash);
        }
        catch (RemoteException e) {
            if (this.recoverConnection(e)) {
                return this.serverHandle.setPassword(saltedSHA256PasswordHash);
            }
            throw e;
        }
    }

    public synchronized boolean canSetPassword() {
        try {
            this.checkServerHandle();
            try {
                return this.serverHandle.canSetPassword();
            }
            catch (RemoteException e) {
                if (this.recoverConnection(e)) {
                    return this.serverHandle.canSetPassword();
                }
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
        return false;
    }

    public ServerInfo getServerInfo() {
        return this.server;
    }

    private void checkServerHandle() throws NotConnectedException {
        if (this.serverHandle == null) {
            throw new NotConnectedException("Not connected to the server");
        }
    }

    boolean hadUnexpectedDisconnect() {
        return this.unexpectedDisconnect;
    }

    public void disconnect() {
        this.disconnect(true);
    }

    private void disconnect(boolean unexpected) {
        if (this.server == null) {
            return;
        }
        this.unexpectedDisconnect = unexpected;
        Msg.warn((Object)this, (Object)("Disconnected from Ghidra Server at " + this.serverInfoStr));
        this.serverHandle = null;
        this.fireStateChanged();
    }
}

