/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.config.java.context;

import static org.hamcrest.CoreMatchers.equalTo;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertThat;

import static test.common.util.Assert.assertBeanDefinitionCount;

import org.junit.Test;

import org.springframework.config.java.annotation.Bean;
import org.springframework.config.java.annotation.Configuration;
import org.springframework.config.java.internal.model.MalformedJavaConfigurationException;

import test.common.beans.TestBean;


/**
 * Unit tests for the {@link JavaConfigApplicationContext} constructor API.
 *
 * @author  Chris Beams
 */
public class JavaConfigApplicationContextConstructionTests {

    @Test(expected = MalformedJavaConfigurationException.class)
    public void testOpenEndedConstructionWithoutSettingClassesOrPackagesIsIllegal() {
        JavaConfigApplicationContext ctx = new JavaConfigApplicationContext();
        ctx.refresh();
        assertBeanDefinitionCount(ctx, 0);
    }

    @Test
    public void testVarargsClassLiterals() {
        JavaConfigApplicationContext ctx = new JavaConfigApplicationContext(ConfigOne.class, ConfigTwo.class);

        int classesInPackage = 2;
        int beansInClasses = 2;

        assertBeanDefinitionCount(ctx, (classesInPackage + beansInClasses));
    }

    @Test
    public void testOpenEndedConstructionWithParentApplicationContextAndSingleClass() {
        JavaConfigApplicationContext ctx = new JavaConfigApplicationContext();
        ctx.setParent(new JavaConfigApplicationContext(ConfigOne.class));
        ctx.addConfigClass(ConfigTwo.class);
        ctx.refresh();

        // beanOne comes from the ConfigOne class
        TestBean beanOne = (TestBean) ctx.getBean("beanOne");
        assertThat(beanOne.getName(), equalTo("one"));

        // beanTwo comes from the ConfigTwo class
        assertNotNull(ctx.getBean("beanTwo"));
    }

    @Test(expected = IllegalStateException.class)
    public void testOpenEndedConstructionWithoutCallingRefreshThrowsException() {
        JavaConfigApplicationContext ctx = new JavaConfigApplicationContext();
        ctx.addConfigClass(ConfigOne.class);
        ctx.getBean("beanOne"); // this should throw - we haven't called refresh
    }

    @Test(expected = IllegalStateException.class)
    public void testSetConfigClassesAfterConstructionThrowsException() {
        JavaConfigApplicationContext ctx = new JavaConfigApplicationContext(ConfigOne.class);
        ctx.addConfigClass(ConfigTwo.class);
    }

    @Test(expected = IllegalStateException.class)
    public void testSetParentAfterConstructionThrowsException() {
        JavaConfigApplicationContext ctx = new JavaConfigApplicationContext(ConfigOne.class);
        ctx.setParent(new JavaConfigApplicationContext(ConfigTwo.class));
    }

    @Test
    public void testConstructionWithMultipleClassNames() {
        JavaConfigApplicationContext ctx = new JavaConfigApplicationContext(ConfigOne.class, ConfigTwo.class);
        assertBeanDefinitionCount(ctx, 4);
    }

    @Configuration
    static class ConfigOne {
        @Bean
        public TestBean beanOne() { return new TestBean("one"); }
    }

    @Configuration
    static class ConfigTwo {
        @Bean
        public TestBean beanTwo() { return new TestBean("two"); }
    }
}
