/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package test.feature.atbean;

import static org.junit.Assert.*;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;

import org.junit.Test;

import org.springframework.aop.support.AopUtils;

import org.springframework.beans.factory.BeanCreationException;

import org.springframework.config.java.annotation.Bean;
import org.springframework.config.java.annotation.Configuration;
import org.springframework.config.java.context.JavaConfigApplicationContext;
import org.springframework.config.java.internal.model.JavaConfigMethod.FinalMethodError;
import org.springframework.config.java.internal.model.MalformedJavaConfigurationException;
import org.springframework.config.java.plugin.aop.AspectJAutoProxy;

import test.common.beans.ITestBean;
import test.common.beans.TestBean;


/**
 * System test for usage errors with {@link Bean} methods.
 *
 * @author  Chris Beams
 */
public class BeanErrorHandlingTests {

    @Test
    public void testInvalidDueToFinalBeanMethod() {
        try {
            new JavaConfigApplicationContext(InvalidDueToFinalBeanMethod.class);
        } catch (MalformedJavaConfigurationException ex) {
            assertTrue(ex.containsError(FinalMethodError.class));
        }
    }

    @Test(expected = MalformedJavaConfigurationException.class)
    public void testInvalidDueToPrivateBeanMethod() {
        // should throw, rejecting private Bean method
        new JavaConfigApplicationContext(InvalidDuePrivateBeanMethod.class);
    }

    @Test(expected = BeanCreationException.class)
    public void testInvalidDueToFinalBeanClass() {
        new JavaConfigApplicationContext(InvalidDueToFinalBeanClass.class);
        // Arguably should spot this earlier
        // should have thrown, rejecting final Bean method
     }

    @Test
    public void testValidWithDynamicProxy() {
        JavaConfigApplicationContext ctx = new JavaConfigApplicationContext(ValidWithDynamicProxies.class);
        ITestBean tb = (ITestBean) ctx.getBean("test");
        assertTrue(AopUtils.isJdkDynamicProxy(tb));
    }

    @Test
    public void testBeanCreationMethodsThatMayThrowExceptions() {
        BeanCreationMethodsThrowExceptions.makeItFail = false;
        JavaConfigApplicationContext ctx = new JavaConfigApplicationContext(BeanCreationMethodsThrowExceptions.class);
        assertNotNull(ctx.getBean("throwsException"));
        assertNotNull(ctx.getBean("throwsThrowable"));
        assertNotNull(ctx.getBean("throwsOtherCheckedException"));
    }

    @Test(expected = BeanCreationException.class)
    public void testBeanCreationMethodsThatDoThrowExceptions() {
        BeanCreationMethodsThrowExceptions.makeItFail = true;
        JavaConfigApplicationContext ctx = new JavaConfigApplicationContext(BeanCreationMethodsThrowExceptions.class);
        ctx.getBean("throwsException");
    }

    @Test
    public void testBeanCreationMethodReturnsNull() {
        JavaConfigApplicationContext ctx =
            new JavaConfigApplicationContext(BeanCreationMethodReturnsNull.class);
        
        TestBean testBean = ctx.getBean(TestBean.class);
        assertNull(testBean);
    }

    @Test(expected = BeanCreationException.class)
    public void testBeanCreationMethodCannotHaveVoidReturn() {
        new JavaConfigApplicationContext(BeanCreationMethodReturnsVoid.class);
    }

    @Configuration
    public static class InvalidDueToFinalBeanMethod {
        @Bean
        public final TestBean testBean() { return new TestBean(); }
    }

    @Configuration
    static class InvalidDuePrivateBeanMethod {
        @Bean
        public Object ok() { return new Object(); }

        @SuppressWarnings("unused")
        @Bean
        private Object notOk() { return new Object(); }
    }


    private static final class FinalTestBean extends TestBean { }

    @AspectJAutoProxy(proxyTargetClass=true)
    @Aspect
    @Configuration
    public static class InvalidDueToFinalBeanClass {
        // Advice causes JavaConfig to attempt to subclass the object returned
        // because FinalTestBean is final, this causes an exception.
        @Before("execution(* get*())")
        public void empty() { }

        @Bean
        public FinalTestBean test() { return new FinalTestBean(); }
    }

    @AspectJAutoProxy
    @Aspect
    @Configuration
    public static class ValidWithDynamicProxies {
        @Before("execution(* get*())")
        public void empty() { }

        @Bean
        public ITestBean test() { return new FinalTestBean(); }
    }

    @Configuration
    public static class BeanCreationMethodsThrowExceptions {
        public static boolean makeItFail;

        @Bean
        public TestBean throwsException() throws Exception {
            if (makeItFail)
                throw new Exception();

            return new TestBean();
        }

        @Bean
        public TestBean throwsThrowable() throws Throwable {
            if (makeItFail)
                throw new Throwable();

            return new TestBean();
        }

        @Bean
        public TestBean throwsOtherCheckedException() throws InterruptedException {
            if (makeItFail)
                throw new InterruptedException();

            return new TestBean();
        }
    }

    @Configuration
    public static class BeanCreationMethodReturnsNull {
        @Bean
        public TestBean returnsNull() { return null; }
    }

    @Configuration
    public static class BeanCreationMethodReturnsVoid {
        @Bean
        public void invalidReturnsVoid() { }
    }

}
