/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package test.feature.autowiring;

import static org.junit.Assert.*;

import org.junit.Assert;
import org.junit.Test;

import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.UnsatisfiedDependencyException;
import org.springframework.beans.factory.annotation.Autowire;
import org.springframework.beans.factory.support.AbstractBeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.DefaultListableBeanFactory;
import org.springframework.beans.factory.support.RootBeanDefinition;

import org.springframework.config.java.annotation.Bean;
import org.springframework.config.java.annotation.Configuration;
import org.springframework.config.java.context.JavaConfigApplicationContext;

import test.common.beans.DependsOnTestBean;
import test.common.beans.TestBean;


/**
 * System tests for {@link Bean#autowire()}
 *
 * @author  Chris Beams
 */
public class BeanAutowireTests {

    @Test
    public void test() {
        BeanFactory bf = new DefaultListableBeanFactory();

        RootBeanDefinition bdFoo = new RootBeanDefinition();
        bdFoo.setBeanClassName(Foo.class.getName());
        bdFoo.setAutowireMode(AbstractBeanDefinition.AUTOWIRE_BY_TYPE);

        RootBeanDefinition bdBar = new RootBeanDefinition();
        bdBar.setBeanClassName(Bar.class.getName());

        ((BeanDefinitionRegistry) bf).registerBeanDefinition("foo", bdFoo);
        ((BeanDefinitionRegistry) bf).registerBeanDefinition("bar", bdBar);

        Foo foo = (Foo) bf.getBean("foo");

        Assert.assertNotNull("bar did not get autowired", foo.getBar());
    }

    @Test
    public void testSJC() {
        JavaConfigApplicationContext ctx = new JavaConfigApplicationContext(MyConfig.class);
        Foo foo = ctx.getBean(Foo.class);

        Assert.assertNotNull("bar did not get autowired", foo.getBar());
    }

    @Test(expected = UnsatisfiedDependencyException.class)
    public void testAutowireAmbiguityIsRejected() {
        try {
            JavaConfigApplicationContext ctx =
                new JavaConfigApplicationContext(InvalidAutowiringConfigurationWithAmbiguity.class);
            ctx.getBean("autowireCandidate");
        } catch (UnsatisfiedDependencyException ex) {
            assertFalse("Useful error message required", ex.getMessage().indexOf("autowireCandidate") == -1);
            throw (ex);
        }
    }

    @Test
    public void testDefaultAutowire() {
        JavaConfigApplicationContext ctx = new JavaConfigApplicationContext(DefaultAutowireConfiguration.class);

        DependsOnTestBean sarah = (DependsOnTestBean) ctx.getBean("sarah");
        assertNotNull("autowiring did not occur: sarah should have TestBean", sarah.getTestBean());
        assertEquals("autowiring error", "adrian", sarah.getTestBean().getName());
    }

    static class Foo {
        private Bar bar;

        public void setBar(Bar bar) { this.bar = bar; }

        public Bar getBar() { return bar; }
    }

    static class Bar { }

    @Configuration(defaultAutowire = Autowire.BY_NAME)
    static class MyConfig {
        @Bean(autowire = Autowire.BY_TYPE)
        public Foo foo() { return new Foo(); }

        @Bean
        public Bar barX() { return new Bar(); }
    }

    @Configuration
    static class InvalidAutowiringConfigurationWithAmbiguity {
        @Bean
        public TestBean testBean() { return new TestBean(); }

        @Bean
        public TestBean nonHiddenTestBean() { return new TestBean("visible"); }

        @Bean(autowire = Autowire.BY_TYPE)
        public DependsOnTestBean autowireCandidate() { return new DependsOnTestBean(); }
    }

    @Configuration(defaultAutowire = Autowire.BY_TYPE)
    public static class DefaultAutowireConfiguration {
        @Bean
        public TestBean adrian() { return new TestBean("adrian", 34); }

        @Bean
        public DependsOnTestBean sarah() { return new DependsOnTestBean(); }
    }


}
