/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.kotlin.components

import kotlin.Boolean
import kotlin.String
import kotlin.Unit
import org.apache.camel.kotlin.CamelDslMarker
import org.apache.camel.kotlin.UriDsl

/**
 * Google BigQuery data warehouse for analytics.
 */
public fun UriDsl.`google-bigquery`(i: GoogleBigqueryUriDsl.() -> Unit) {
  GoogleBigqueryUriDsl(this).apply(i)
}

@CamelDslMarker
public class GoogleBigqueryUriDsl(
  it: UriDsl,
) {
  private val it: UriDsl

  init {
    this.it = it
    this.it.component("google-bigquery")
  }

  private var projectId: String = ""

  private var datasetId: String = ""

  private var tableId: String = ""

  /**
   * Google Cloud Project Id
   */
  public fun projectId(projectId: String) {
    this.projectId = projectId
    it.url("$projectId:$datasetId:$tableId")
  }

  /**
   * BigQuery Dataset Id
   */
  public fun datasetId(datasetId: String) {
    this.datasetId = datasetId
    it.url("$projectId:$datasetId:$tableId")
  }

  /**
   * BigQuery table id
   */
  public fun tableId(tableId: String) {
    this.tableId = tableId
    it.url("$projectId:$datasetId:$tableId")
  }

  /**
   * ConnectionFactory to obtain connection to Bigquery Service. If not provided the default one
   * will be used
   */
  public fun connectionFactory(connectionFactory: String) {
    it.property("connectionFactory", connectionFactory)
  }

  /**
   * Field name to use as insert id
   */
  public fun useAsInsertId(useAsInsertId: String) {
    it.property("useAsInsertId", useAsInsertId)
  }

  /**
   * Whether the producer should be started lazy (on the first message). By starting lazy you can
   * use this to allow CamelContext and routes to startup in situations where a producer may otherwise
   * fail during starting and cause the route to fail being started. By deferring this startup to be
   * lazy then the startup failure can be handled during routing messages via Camel's routing error
   * handlers. Beware that when the first message is processed then creating and starting the producer
   * may take a little time and prolong the total processing time of the processing.
   */
  public fun lazyStartProducer(lazyStartProducer: String) {
    it.property("lazyStartProducer", lazyStartProducer)
  }

  /**
   * Whether the producer should be started lazy (on the first message). By starting lazy you can
   * use this to allow CamelContext and routes to startup in situations where a producer may otherwise
   * fail during starting and cause the route to fail being started. By deferring this startup to be
   * lazy then the startup failure can be handled during routing messages via Camel's routing error
   * handlers. Beware that when the first message is processed then creating and starting the producer
   * may take a little time and prolong the total processing time of the processing.
   */
  public fun lazyStartProducer(lazyStartProducer: Boolean) {
    it.property("lazyStartProducer", lazyStartProducer.toString())
  }

  /**
   * Service account key in json format to authenticate an application as a service account to
   * google cloud platform
   */
  public fun serviceAccountKey(serviceAccountKey: String) {
    it.property("serviceAccountKey", serviceAccountKey)
  }
}
