import { FeelSyntacticSymbolNature } from "./FeelSyntacticSymbolNature";
import { Expression } from "./Expression";
import { BuiltInTypes } from "./BuiltInTypes";
export class IdentifiersRepository {
    constructor(dmnDefinitions, externalDefinitions) {
        this.dmnDefinitions = dmnDefinitions;
        this.externalDefinitions = externalDefinitions;
        this._dataTypes = new Map([
            [BuiltInTypes.Number.name, BuiltInTypes.Number],
            [BuiltInTypes.Boolean.name, BuiltInTypes.Boolean],
            [BuiltInTypes.String.name, BuiltInTypes.String],
            [BuiltInTypes.DaysAndTimeDuration.name, BuiltInTypes.DaysAndTimeDuration],
            [BuiltInTypes.DateAndTime.name, BuiltInTypes.DateAndTime],
            [BuiltInTypes.YearsAndMonthsDuration.name, BuiltInTypes.YearsAndMonthsDuration],
            [BuiltInTypes.Time.name, BuiltInTypes.Time],
            [BuiltInTypes.Date.name, BuiltInTypes.Date],
        ]);
        this._identifiersContextIndexedByUuid = new Map();
        this._expressionsIndexedByUuid = new Map();
        this._dataTypeIndexedByUuid = new Map();
        this._importedIdentifiers = new Map();
        this._importedDataTypes = new Map();
        this.loadImportedIdentifiers(dmnDefinitions, externalDefinitions);
        this.currentIdentifierNamePrefix = "";
        this.currentUuidPrefix = "";
        this.loadIdentifiers(dmnDefinitions);
    }
    get identifiersContextIndexedByUuid() {
        return this._identifiersContextIndexedByUuid;
    }
    get expressionsIndexedByUuid() {
        return this._expressionsIndexedByUuid;
    }
    get dataTypeIndexedByUuid() {
        return this._dataTypeIndexedByUuid;
    }
    get importedDataTypes() {
        return this._importedDataTypes;
    }
    get importedIdentifiers() {
        return this._importedIdentifiers;
    }
    get identifiers() {
        return this._identifiersContextIndexedByUuid;
    }
    get dataTypes() {
        return this._dataTypes;
    }
    get expressions() {
        return this._expressionsIndexedByUuid;
    }
    reload() {
        this._identifiersContextIndexedByUuid.clear();
        this._expressionsIndexedByUuid.clear();
        this._dataTypeIndexedByUuid.clear();
        this._importedIdentifiers.clear();
        this._importedDataTypes.clear();
        this.loadImportedIdentifiers(this.dmnDefinitions, this.externalDefinitions);
        this.currentIdentifierNamePrefix = "";
        this.currentUuidPrefix = "";
        this.loadIdentifiers(this.dmnDefinitions);
    }
    createDataTypes(definitions) {
        var _a;
        (_a = definitions.itemDefinition) === null || _a === void 0 ? void 0 : _a.forEach((itemDefinition) => {
            var _a;
            const dataType = this.createDataType(itemDefinition);
            this._dataTypeIndexedByUuid.set(dataType.uuid, dataType);
            this.addImportedDataType(dataType);
            (_a = itemDefinition.itemComponent) === null || _a === void 0 ? void 0 : _a.forEach((itemComponent) => {
                const innerType = this.createInnerType(itemComponent);
                this._dataTypeIndexedByUuid.set(innerType.uuid, innerType);
                this.addImportedDataType(innerType);
                dataType.properties.set(innerType.name, innerType);
            });
            this.dataTypes.set(dataType.name, dataType);
        });
    }
    addImportedDataType(dataType) {
        var _a;
        if (this.currentIdentifierNamePrefix.length != 0) {
            if (!this._importedDataTypes.has(this.currentIdentifierNamePrefix)) {
                this._importedDataTypes.set(this.currentIdentifierNamePrefix, []);
            }
            (_a = this._importedDataTypes.get(this.currentIdentifierNamePrefix)) === null || _a === void 0 ? void 0 : _a.push(dataType);
        }
    }
    loadIdentifiersFromDefinitions(definitions) {
        var _a;
        (_a = definitions.drgElement) === null || _a === void 0 ? void 0 : _a.forEach((drg) => {
            switch (drg.__$$element) {
                case "decision":
                    this.loadIdentifiersFromDecision(drg);
                    break;
                case "inputData":
                    this.loadIdentifiersFromInputData(drg);
                    break;
                case "businessKnowledgeModel":
                    this.loadIdentifiersFromBkm(drg);
                    break;
                case "decisionService":
                    this.loadIdentifiersFromDecisionService(drg);
                    break;
                default:
                    break;
            }
        });
    }
    loadIdentifiersFromInputData(drg) {
        var _a, _b;
        this.addIdentifierContext({
            uuid: (_a = drg["@_id"]) !== null && _a !== void 0 ? _a : "",
            identifierDefinedByTheContext: drg["@_name"],
            kind: FeelSyntacticSymbolNature.GlobalVariable,
            parentContext: undefined,
            typeRef: (_b = drg.variable) === null || _b === void 0 ? void 0 : _b["@_typeRef"],
            applyValueToSource: (value) => {
                drg["@_name"] = value;
            },
            applyTypeRefToSource: (value) => {
                if (drg.variable) {
                    if (typeof value === "string") {
                        drg.variable["@_typeRef"] = value;
                    }
                    else {
                        drg.variable["@_typeRef"] = value === null || value === void 0 ? void 0 : value.typeRef;
                    }
                }
            },
        });
    }
    loadIdentifiersFromDecisionService(drg) {
        var _a, _b;
        this.addIdentifierContext({
            uuid: (_a = drg["@_id"]) !== null && _a !== void 0 ? _a : "",
            identifierDefinedByTheContext: drg["@_name"],
            kind: FeelSyntacticSymbolNature.Invocable,
            parentContext: undefined,
            typeRef: (_b = drg.variable) === null || _b === void 0 ? void 0 : _b["@_typeRef"],
            applyValueToSource: (value) => {
                drg["@_name"] = value;
            },
            applyTypeRefToSource: (value) => {
                if (drg.variable) {
                    if (typeof value === "string") {
                        drg.variable["@_typeRef"] = value;
                    }
                    else {
                        drg.variable["@_typeRef"] = value === null || value === void 0 ? void 0 : value.typeRef;
                    }
                }
            },
        });
    }
    loadIdentifiersFromBkm(drg) {
        var _a, _b, _c, _d;
        const parent = this.addIdentifierContext({
            uuid: (_a = drg["@_id"]) !== null && _a !== void 0 ? _a : "",
            identifierDefinedByTheContext: drg["@_name"],
            kind: FeelSyntacticSymbolNature.Invocable,
            parentContext: undefined,
            typeRef: (_b = drg.variable) === null || _b === void 0 ? void 0 : _b["@_typeRef"],
            applyValueToSource: (value) => {
                drg["@_name"] = value;
            },
            applyTypeRefToSource: (value) => {
                if (drg.variable) {
                    if (typeof value === "string") {
                        drg.variable["@_typeRef"] = value;
                    }
                    else {
                        drg.variable["@_typeRef"] = value === null || value === void 0 ? void 0 : value.typeRef;
                    }
                }
            },
        });
        if (drg.encapsulatedLogic) {
            let parentElement = parent;
            if (drg.encapsulatedLogic.formalParameter) {
                for (const parameter of drg.encapsulatedLogic.formalParameter) {
                    parentElement = this.addIdentifierContext({
                        uuid: (_c = parameter["@_id"]) !== null && _c !== void 0 ? _c : "",
                        identifierDefinedByTheContext: (_d = parameter["@_name"]) !== null && _d !== void 0 ? _d : "<parameter>",
                        kind: FeelSyntacticSymbolNature.Parameter,
                        parentContext: parentElement,
                        applyValueToSource: (value) => {
                            parameter["@_name"] = value;
                        },
                    });
                }
            }
            if (drg.encapsulatedLogic.expression) {
                this.addInnerExpression(parentElement, drg.encapsulatedLogic.expression);
            }
        }
    }
    loadIdentifiersFromDecision(drg) {
        var _a, _b;
        const parent = this.addIdentifierContext({
            uuid: (_a = drg["@_id"]) !== null && _a !== void 0 ? _a : "",
            identifierDefinedByTheContext: drg["@_name"],
            kind: FeelSyntacticSymbolNature.InvisibleVariables,
            parentContext: undefined,
            typeRef: (_b = drg.variable) === null || _b === void 0 ? void 0 : _b["@_typeRef"],
            applyValueToSource: (value) => {
                drg["@_name"] = value;
            },
            applyTypeRefToSource: (value) => {
                if (drg.variable) {
                    if (typeof value === "string") {
                        drg.variable["@_typeRef"] = value;
                    }
                    else {
                        drg.variable["@_typeRef"] = value === null || value === void 0 ? void 0 : value.typeRef;
                    }
                }
            },
        });
        if (drg.informationRequirement) {
            for (const requirement of drg.informationRequirement) {
                this.addInputVariable(parent, requirement);
            }
        }
        if (drg.knowledgeRequirement) {
            for (const knowledgeRequirement of drg.knowledgeRequirement) {
                this.addInputVariableFromKnowledge(parent, knowledgeRequirement);
            }
        }
        if (drg.expression) {
            this.addInnerExpression(parent, drg.expression);
        }
    }
    addIdentifierContext(args) {
        var _a;
        const variableContext = this.createIdentifierContext(this.buildIdentifierUuid(args.uuid), this.buildName(args.identifierDefinedByTheContext), args.kind, args.parentContext, args.typeRef, args.allowDynamicVariables, args.applyValueToSource, args.applyTypeRefToSource);
        if (this.currentIdentifierNamePrefix.length != 0) {
            if (!this._importedIdentifiers.has(this.currentIdentifierNamePrefix)) {
                this._importedIdentifiers.set(this.currentIdentifierNamePrefix, []);
            }
            (_a = this._importedIdentifiers.get(this.currentIdentifierNamePrefix)) === null || _a === void 0 ? void 0 : _a.push(variableContext);
        }
        this._identifiersContextIndexedByUuid.set(this.buildIdentifierUuid(args.uuid), variableContext);
        return variableContext;
    }
    createIdentifierContext(uuid, identifierDefinedByTheContext, variableType, parent, typeRef, allowDynamicVariables, applyValueToSource, applyTypeRefToSource) {
        return {
            uuid: uuid,
            children: new Map(),
            parent: parent,
            inputIdentifiers: new Array(),
            allowDynamicVariables: allowDynamicVariables,
            identifier: {
                value: identifierDefinedByTheContext,
                feelSyntacticSymbolNature: variableType,
                typeRef: this.getTypeRef(typeRef),
                expressionsThatUseTheIdentifier: new Map(),
                applyValueToSource() {
                    if (applyValueToSource) {
                        applyValueToSource(this.value);
                    }
                },
                applyTypeRefToSource() {
                    if (applyTypeRefToSource) {
                        applyTypeRefToSource(this.typeRef);
                    }
                },
            },
        };
    }
    getTypeRef(typeRef) {
        return this.dataTypes.has(typeRef !== null && typeRef !== void 0 ? typeRef : "") ? this.dataTypes.get(typeRef !== null && typeRef !== void 0 ? typeRef : "") : typeRef;
    }
    createDataType(itemDefinition) {
        var _a, _b, _c;
        const name = this.buildName(itemDefinition["@_name"]);
        const dataType = {
            uuid: (_a = itemDefinition["@_id"]) !== null && _a !== void 0 ? _a : "datatype_uuid",
            name: name,
            properties: new Map(),
            typeRef: (_c = (_b = itemDefinition["typeRef"]) === null || _b === void 0 ? void 0 : _b.__$$text) !== null && _c !== void 0 ? _c : itemDefinition["@_name"],
            source: {
                value: name,
                feelSyntacticSymbolNature: FeelSyntacticSymbolNature.GlobalVariable,
                expressionsThatUseTheIdentifier: new Map(),
            },
        };
        return dataType;
    }
    createInnerType(itemComponent) {
        var _a, _b, _c;
        const dataType = {
            uuid: (_a = itemComponent["@_id"]) !== null && _a !== void 0 ? _a : "item_uuid",
            name: itemComponent["@_name"],
            properties: this.buildProperties(itemComponent),
            typeRef: (_c = (_b = itemComponent["typeRef"]) === null || _b === void 0 ? void 0 : _b.__$$text) !== null && _c !== void 0 ? _c : itemComponent["@_name"],
            source: {
                value: itemComponent["@_name"],
                feelSyntacticSymbolNature: FeelSyntacticSymbolNature.GlobalVariable,
                expressionsThatUseTheIdentifier: new Map(),
            },
        };
        return dataType;
    }
    buildProperties(itemComponent) {
        var _a;
        const properties = new Map();
        (_a = itemComponent.itemComponent) === null || _a === void 0 ? void 0 : _a.forEach((def) => {
            var _a, _b, _c;
            const property = {
                uuid: (_a = def["@_id"]) !== null && _a !== void 0 ? _a : "root_property",
                name: def["@_name"],
                properties: this.buildProperties(def),
                typeRef: (_c = (_b = def["typeRef"]) === null || _b === void 0 ? void 0 : _b.__$$text) !== null && _c !== void 0 ? _c : def["@_name"],
                source: {
                    value: def["@_name"],
                    feelSyntacticSymbolNature: FeelSyntacticSymbolNature.GlobalVariable,
                    expressionsThatUseTheIdentifier: new Map(),
                },
            };
            this._dataTypeIndexedByUuid.set(property.uuid, property);
            properties.set(property.name, property);
        });
        return properties;
    }
    addExpression(parent, element) {
        var _a;
        const id = (_a = element["@_id"]) !== null && _a !== void 0 ? _a : "";
        const expression = new Expression(id, element);
        this._expressionsIndexedByUuid.set(id, expression);
        this.addIdentifierContext({
            uuid: id,
            identifierDefinedByTheContext: "",
            kind: FeelSyntacticSymbolNature.LocalVariable,
            parentContext: parent,
        });
    }
    addInvocation(parent, element) {
        if (element.binding) {
            for (const bindingElement of element.binding) {
                if (bindingElement.expression) {
                    this.addInnerExpression(parent, bindingElement.expression);
                }
            }
        }
    }
    addContext(parent, element) {
        let parentNode = parent;
        if (element.contextEntry) {
            for (const innerEntry of element.contextEntry) {
                parentNode = this.addContextEntry(parentNode, innerEntry);
            }
        }
    }
    addContextEntry(parentNode, contextEntry) {
        var _a, _b, _c, _d, _e;
        const variableNode = this.addIdentifierContext({
            uuid: (_b = (_a = contextEntry.variable) === null || _a === void 0 ? void 0 : _a["@_id"]) !== null && _b !== void 0 ? _b : "",
            identifierDefinedByTheContext: (_d = (_c = contextEntry.variable) === null || _c === void 0 ? void 0 : _c["@_name"]) !== null && _d !== void 0 ? _d : "",
            kind: FeelSyntacticSymbolNature.LocalVariable,
            parentContext: parentNode,
            typeRef: (_e = contextEntry.variable) === null || _e === void 0 ? void 0 : _e["@_typeRef"],
            applyValueToSource: (value) => {
                if (contextEntry.variable) {
                    contextEntry.variable["@_name"] = value;
                }
            },
            applyTypeRefToSource: (value) => {
                if (contextEntry.variable) {
                    if (typeof value === "string") {
                        contextEntry.variable["@_typeRef"] = value;
                    }
                    else {
                        contextEntry.variable["@_typeRef"] = value === null || value === void 0 ? void 0 : value.typeRef;
                    }
                }
            },
        });
        parentNode.children.set(variableNode.uuid, variableNode);
        if (contextEntry.expression) {
            if (contextEntry.expression.__$$element) {
                this.addInnerExpression(parentNode, contextEntry.expression);
            }
        }
        return variableNode;
    }
    addFunctionDefinition(parent, element) {
        var _a, _b;
        let parentElement = parent;
        if (element.formalParameter) {
            for (const parameter of element.formalParameter) {
                parentElement = this.addIdentifierContext({
                    uuid: (_a = parameter["@_id"]) !== null && _a !== void 0 ? _a : "",
                    identifierDefinedByTheContext: (_b = parameter["@_name"]) !== null && _b !== void 0 ? _b : "<parameter>",
                    kind: FeelSyntacticSymbolNature.Parameter,
                    parentContext: parentElement,
                    applyValueToSource: (value) => {
                        parameter["@_name"] = value;
                    },
                });
            }
        }
        if (element.expression) {
            this.addInnerExpression(parentElement, element.expression);
        }
    }
    addRelation(parent, element) {
        if (element.row) {
            for (const rowElement of element.row) {
                if (rowElement.expression) {
                    for (const expression of rowElement.expression) {
                        this.addInnerExpression(parent, expression);
                    }
                }
            }
        }
    }
    addList(parent, element) {
        if (element.expression) {
            for (const expression of element.expression) {
                if (expression) {
                    this.addInnerExpression(parent, expression);
                }
            }
        }
    }
    addConditional(parent, element) {
        var _a, _b, _c;
        if ((_a = element.if) === null || _a === void 0 ? void 0 : _a.expression) {
            this.addInnerExpression(parent, element.if.expression);
        }
        if ((_b = element.then) === null || _b === void 0 ? void 0 : _b.expression) {
            this.addInnerExpression(parent, element.then.expression);
        }
        if ((_c = element.else) === null || _c === void 0 ? void 0 : _c.expression) {
            this.addInnerExpression(parent, element.else.expression);
        }
    }
    addIterable(parent, expression) {
        const localParent = this.addIteratorVariable(parent, expression);
        if (expression.in.expression) {
            this.addInnerExpression(localParent, expression.in.expression);
        }
        if (expression.satisfies.expression) {
            this.addInnerExpression(localParent, expression.satisfies.expression);
        }
    }
    addFor(parent, expression) {
        const localParent = this.addIteratorVariable(parent, expression);
        if (expression.return.expression) {
            this.addInnerExpression(localParent, expression.return.expression);
        }
        if (expression.in.expression) {
            this.addInnerExpression(localParent, expression.in.expression);
        }
    }
    addFilterVariable(parent, expression) {
        var _a;
        let type = undefined;
        if (expression.in.expression) {
            type = expression.in.expression["@_typeRef"];
        }
        return this.addIdentifierContext({
            uuid: (_a = expression["@_id"]) !== null && _a !== void 0 ? _a : "",
            identifierDefinedByTheContext: "item",
            kind: FeelSyntacticSymbolNature.LocalVariable,
            parentContext: parent,
            typeRef: type,
            allowDynamicVariables: true,
        });
    }
    addIteratorVariable(parent, expression) {
        var _a;
        let localParent = parent;
        if (expression["@_iteratorVariable"]) {
            let type = undefined;
            if (expression.in.expression) {
                type = expression.in.expression["@_typeRef"];
            }
            localParent = this.addIdentifierContext({
                uuid: (_a = expression["@_id"]) !== null && _a !== void 0 ? _a : "",
                identifierDefinedByTheContext: expression["@_iteratorVariable"],
                kind: FeelSyntacticSymbolNature.LocalVariable,
                parentContext: parent,
                typeRef: type,
                allowDynamicVariables: true,
                applyValueToSource: (value) => {
                    expression["@_iteratorVariable"] = value;
                },
                applyTypeRefToSource: (value) => {
                    if (expression.in.expression) {
                        if (typeof value === "string") {
                            expression.in.expression["@_typeRef"] = value;
                        }
                        else {
                            expression.in.expression["@_typeRef"] = value === null || value === void 0 ? void 0 : value.typeRef;
                        }
                    }
                },
            });
        }
        return localParent;
    }
    addFilter(parent, expression) {
        if (expression.in.expression) {
            this.addInnerExpression(parent, expression.in.expression);
        }
        const localParent = this.addFilterVariable(parent, expression);
        if (expression.match.expression) {
            this.addInnerExpression(localParent, expression.match.expression);
        }
    }
    addInnerExpression(parent, expression) {
        switch (expression.__$$element) {
            case "literalExpression":
                this.addExpression(parent, expression);
                break;
            case "invocation":
                this.addInvocation(parent, expression);
                break;
            case "decisionTable":
                this.addDecisionTable(parent, expression);
                break;
            case "context":
                this.addContext(parent, expression);
                break;
            case "functionDefinition":
                this.addFunctionDefinition(parent, expression);
                break;
            case "relation":
                this.addRelation(parent, expression);
                break;
            case "list":
                this.addList(parent, expression);
                break;
            case "conditional":
                this.addConditional(parent, expression);
                break;
            case "every":
            case "some":
                this.addIterable(parent, expression);
                break;
            case "for":
                this.addFor(parent, expression);
                break;
            case "filter":
                this.addFilter(parent, expression);
                break;
            default:
        }
    }
    addDecisionTableEntryNode(parent, entryNode) {
        var _a;
        const ruleInputElementNode = this.addIdentifierContext({
            uuid: (_a = entryNode["@_id"]) !== null && _a !== void 0 ? _a : "",
            identifierDefinedByTheContext: "",
            kind: FeelSyntacticSymbolNature.LocalVariable,
            parentContext: parent,
        });
        parent.children.set(ruleInputElementNode.uuid, ruleInputElementNode);
        this.addExpression(parent, entryNode);
    }
    addDecisionTableInputEntryNode(parent, inputEntryNode) {
        var _a;
        const identifierContext = this.addIdentifierContext({
            uuid: (_a = inputEntryNode["@_id"]) !== null && _a !== void 0 ? _a : "",
            identifierDefinedByTheContext: "",
            kind: FeelSyntacticSymbolNature.LocalVariable,
            parentContext: parent,
        });
        parent.children.set(identifierContext.uuid, identifierContext);
        this.addExpression(parent, inputEntryNode.inputExpression);
    }
    addDecisionTable(parent, decisionTable) {
        var _a, _b, _c;
        const addedIdentifierContext = this.addIdentifierContext({
            uuid: (_a = decisionTable["@_id"]) !== null && _a !== void 0 ? _a : "",
            identifierDefinedByTheContext: "",
            kind: FeelSyntacticSymbolNature.LocalVariable,
            parentContext: parent,
        });
        parent.children.set(addedIdentifierContext.uuid, addedIdentifierContext);
        if (decisionTable.rule) {
            for (const ruleElement of decisionTable.rule) {
                (_b = ruleElement.inputEntry) === null || _b === void 0 ? void 0 : _b.forEach((inputElement) => this.addDecisionTableEntryNode(parent, inputElement));
                (_c = ruleElement.outputEntry) === null || _c === void 0 ? void 0 : _c.forEach((outputElement) => this.addDecisionTableEntryNode(parent, outputElement));
            }
        }
        if (decisionTable.input) {
            for (const inputClause of decisionTable.input) {
                this.addDecisionTableInputEntryNode(addedIdentifierContext, inputClause);
            }
        }
        this.addIdentifierContext({
            uuid: addedIdentifierContext.uuid,
            identifierDefinedByTheContext: "",
            kind: FeelSyntacticSymbolNature.LocalVariable,
            parentContext: parent,
            applyTypeRefToSource: (value) => {
                if (typeof value === "string") {
                    decisionTable["@_typeRef"] = value;
                }
                else {
                    decisionTable["@_typeRef"] = value === null || value === void 0 ? void 0 : value.typeRef;
                }
            },
        });
    }
    addInputVariable(parent, requirement) {
        var _a, _b;
        if (requirement.requiredDecision) {
            parent.inputIdentifiers.push((_a = requirement.requiredDecision["@_href"]) === null || _a === void 0 ? void 0 : _a.replace("#", ""));
        }
        else if (requirement.requiredInput) {
            parent.inputIdentifiers.push((_b = requirement.requiredInput["@_href"]) === null || _b === void 0 ? void 0 : _b.replace("#", ""));
        }
    }
    addInputVariableFromKnowledge(parent, knowledgeRequirement) {
        var _a;
        if (knowledgeRequirement.requiredKnowledge) {
            parent.inputIdentifiers.push((_a = knowledgeRequirement.requiredKnowledge["@_href"]) === null || _a === void 0 ? void 0 : _a.replace("#", ""));
        }
    }
    buildIdentifierUuid(uuid) {
        if (this.currentUuidPrefix.length != 0) {
            return this.currentUuidPrefix + uuid;
        }
        return uuid;
    }
    buildName(name) {
        if (this.currentIdentifierNamePrefix.length != 0) {
            return `${this.currentIdentifierNamePrefix}.${name}`;
        }
        return name;
    }
    loadIdentifiers(dmnDefinitions) {
        this.createDataTypes(dmnDefinitions);
        this.loadIdentifiersFromDefinitions(dmnDefinitions);
    }
    loadImportedIdentifiers(dmnDefinitions, externalDefinitions) {
        if (!(dmnDefinitions.import && externalDefinitions)) {
            return;
        }
        for (const dmnImport of dmnDefinitions.import.filter((imp) => externalDefinitions.has(imp["@_namespace"]))) {
            this.currentIdentifierNamePrefix = dmnImport["@_name"];
            this.currentUuidPrefix = dmnImport["@_namespace"];
            const externalDef = externalDefinitions.get(dmnImport["@_namespace"]);
            if (externalDef) {
                this.loadIdentifiers(externalDef.definitions);
            }
        }
    }
}
//# sourceMappingURL=IdentifiersRepository.js.map