/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.serializer;

import static org.apache.juneau.commons.utils.CollectionUtils.*;
import static org.junit.jupiter.api.Assertions.*;

import java.nio.charset.Charset;
import java.util.function.*;

import org.apache.juneau.*;
import org.apache.juneau.commons.reflect.*;
import org.apache.juneau.json.*;
import org.apache.juneau.msgpack.*;
import org.apache.juneau.serializer.annotation.*;
import org.apache.juneau.svl.*;
import org.junit.jupiter.api.*;

/**
 * Tests the @SerializerConfig annotation.
 */
class SerializerConfigAnnotation_Test extends TestBase {

	private static void check(String expected, Object o) {
		assertEquals(expected, TO_STRING.apply(o));
	}

	private static final Function<Object,String> TO_STRING = t -> {
		if (t == null)
			return null;
		if (t instanceof AA)
			return "AA";
		return t.toString();
	};

	static VarResolverSession sr = VarResolver.create().vars(XVar.class).build().createSession();

	//-----------------------------------------------------------------------------------------------------------------
	// Basic tests
	//-----------------------------------------------------------------------------------------------------------------

	public static class AA extends SerializerListener {}

	@SerializerConfig(
		addBeanTypes="$X{true}",
		addRootType="$X{true}",
		binaryFormat="$X{HEX}",
		detectRecursions="$X{true}",
		ignoreRecursions="$X{true}",
		initialDepth="$X{1}",
		listener=AA.class,
		maxDepth="$X{1}",
		maxIndent="$X{1}",
		quoteChar="$X{'}",
		sortCollections="$X{true}",
		sortMaps="$X{true}",
		trimEmptyCollections="$X{true}",
		trimEmptyMaps="$X{true}",
		keepNullProperties="$X{false}",
		trimStrings="$X{true}",
		uriContext="{}",
		uriRelativity="$X{RESOURCE}",
		uriResolution="$X{ABSOLUTE}",
		useWhitespace="$X{true}"
	)
	static class A {}
	static ClassInfo a = ClassInfo.of(A.class);

	@Test void a01_basicWriterSerializer() {
		var al = AnnotationWorkList.of(sr, rstream(a.getAnnotations()));
		var x = JsonSerializer.create().apply(al).build().getSession();
		check("true", ((SerializerSession)x).isAddBeanTypes());
		check("true", x.isAddRootType());
		check("true", x.isDetectRecursions());
		check("true", x.isIgnoreRecursions());
		check("1", x.getInitialDepth());
		check("AA", x.getListener());
		check("1", x.getMaxDepth());
		check("1", x.getMaxIndent());
		check("'", x.getQuoteChar());
		check("true", x.isSortCollections());
		check("true", x.isSortMaps());
		check("true", x.isTrimEmptyCollections());
		check("true", x.isTrimEmptyMaps());
		check("false", x.isKeepNullProperties());
		check("true", x.isTrimStrings());
		check("{aContextRoot=/,aPathInfo=/,aServletPath=/,rContextRoot=/,rPath=/,rResource=/}", x.getUriContext());
		check("RESOURCE", x.getUriRelativity());
		check("ABSOLUTE", x.getUriResolution());
		check("true", x.isUseWhitespace());
	}

	@Test void a02_basicOutputStreamSerializer() {
		var al = AnnotationWorkList.of(sr, rstream(a.getAnnotations()));
		var x = MsgPackSerializer.create().apply(al).build().getSession();
		check("true", ((SerializerSession)x).isAddBeanTypes());
		check("true", x.isAddRootType());
		check("HEX", x.getBinaryFormat());
		check("true", x.isDetectRecursions());
		check("true", x.isIgnoreRecursions());
		check("1", x.getInitialDepth());
		check("AA", x.getListener());
		check("1", x.getMaxDepth());
		check("true", x.isSortCollections());
		check("true", x.isSortMaps());
		check("true", x.isTrimEmptyCollections());
		check("true", x.isTrimEmptyMaps());
		check("false", x.isKeepNullProperties());
		check("true", x.isTrimStrings());
		check("{aContextRoot=/,aPathInfo=/,aServletPath=/,rContextRoot=/,rPath=/,rResource=/}", x.getUriContext());
		check("RESOURCE", x.getUriRelativity());
		check("ABSOLUTE", x.getUriResolution());
	}

	//-----------------------------------------------------------------------------------------------------------------
	// Annotation with no values.
	//-----------------------------------------------------------------------------------------------------------------

	@SerializerConfig()
	static class B {}
	static ClassInfo b = ClassInfo.of(B.class);

	@Test void b01_noValuesWriterSerializer() {
		var al = AnnotationWorkList.of(sr, rstream(b.getAnnotations()));
		var x = JsonSerializer.create().apply(al).build().getSession();
		check("false", ((SerializerSession)x).isAddBeanTypes());
		check("false", x.isAddRootType());
		check(null, x.getListener());
		check("100", x.getMaxIndent());
		check("\"", x.getQuoteChar());
		check("false", x.isSortCollections());
		check("false", x.isSortMaps());
		check("false", x.isTrimEmptyCollections());
		check("false", x.isTrimEmptyMaps());
		check("false", x.isKeepNullProperties());
		check("false", x.isTrimStrings());
		check("{aContextRoot=/,aPathInfo=/,aServletPath=/,rContextRoot=/,rPath=/,rResource=/}", x.getUriContext());
		check("RESOURCE", x.getUriRelativity());
		check("NONE", x.getUriResolution());
		check("false", x.isUseWhitespace());
	}

	@Test void b02_noValuesOutputStreamSerializer() {
		var al = AnnotationWorkList.of(sr, rstream(b.getAnnotations()));
		var x = MsgPackSerializer.create().apply(al).build().getSession();
		check("false", ((SerializerSession)x).isAddBeanTypes());
		check("false", x.isAddRootType());
		check("HEX", x.getBinaryFormat());
		check(null, x.getListener());
		check("false", x.isSortCollections());
		check("false", x.isSortMaps());
		check("false", x.isTrimEmptyCollections());
		check("false", x.isTrimEmptyMaps());
		check("false", x.isKeepNullProperties());
		check("false", x.isTrimStrings());
		check("{aContextRoot=/,aPathInfo=/,aServletPath=/,rContextRoot=/,rPath=/,rResource=/}", x.getUriContext());
		check("RESOURCE", x.getUriRelativity());
		check("NONE", x.getUriResolution());
	}

	//-----------------------------------------------------------------------------------------------------------------
	// No annotation.
	//-----------------------------------------------------------------------------------------------------------------

	static class C {}
	static ClassInfo c = ClassInfo.of(C.class);

	@Test void c01_noAnnotationWriterSerializer() {
		var al = AnnotationWorkList.of(sr, rstream(c.getAnnotations()));
		var x = JsonSerializer.create().apply(al).build().getSession();
		check("false", ((SerializerSession)x).isAddBeanTypes());
		check("false", x.isAddRootType());
		check(null, x.getListener());
		check("100", x.getMaxIndent());
		check("\"", x.getQuoteChar());
		check("false", x.isSortCollections());
		check("false", x.isSortMaps());
		check("false", x.isTrimEmptyCollections());
		check("false", x.isTrimEmptyMaps());
		check("false", x.isKeepNullProperties());
		check("false", x.isTrimStrings());
		check("{aContextRoot=/,aPathInfo=/,aServletPath=/,rContextRoot=/,rPath=/,rResource=/}", x.getUriContext());
		check("RESOURCE", x.getUriRelativity());
		check("NONE", x.getUriResolution());
		check("false", x.isUseWhitespace());
	}

	@Test void c02_noAnnotationOutputStreamSerializer() {
		var al = AnnotationWorkList.of(sr, rstream(c.getAnnotations()));
		var x = MsgPackSerializer.create().apply(al).build().getSession();
		check("false", ((SerializerSession)x).isAddBeanTypes());
		check("false", x.isAddRootType());
		check("HEX", x.getBinaryFormat());
		check(null, x.getListener());
		check("false", x.isSortCollections());
		check("false", x.isSortMaps());
		check("false", x.isTrimEmptyCollections());
		check("false", x.isTrimEmptyMaps());
		check("false", x.isKeepNullProperties());
		check("false", x.isTrimStrings());
		check("{aContextRoot=/,aPathInfo=/,aServletPath=/,rContextRoot=/,rPath=/,rResource=/}", x.getUriContext());
		check("RESOURCE", x.getUriRelativity());
		check("NONE", x.getUriResolution());
	}

	//-----------------------------------------------------------------------------------------------------------------
	// Error cases - invalid values
	//-----------------------------------------------------------------------------------------------------------------

	@SerializerConfig(
		quoteChar="$X{ab}"
	)
	static class D {}
	static ClassInfo d = ClassInfo.of(D.class);

	@Test void d01_invalidQuoteChar_throwsException() {
		var al = AnnotationWorkList.of(sr, rstream(d.getAnnotations()));
		assertThrows(ConfigException.class, () -> {
			JsonSerializer.create().apply(al).build();
		});
	}

	@SerializerConfig(
		fileCharset="$X{UTF-8}",
		streamCharset="$X{ISO-8859-1}"
	)
	static class E {}
	static ClassInfo e = ClassInfo.of(E.class);

	@Test void d02_charsetWithNonDefaultValue() {
		var al = AnnotationWorkList.of(sr, rstream(e.getAnnotations()));
		var x = JsonSerializer.create().apply(al).build().getSession();
		check("UTF-8", x.getFileCharset().name());
		check("ISO-8859-1", x.getStreamCharset().name());
	}

	@SerializerConfig(
		fileCharset="$X{default}",
		streamCharset="$X{DEFAULT}"
	)
	static class E2 {}
	static ClassInfo e2 = ClassInfo.of(E2.class);

	@Test void d02b_charsetWithDefaultValue() {
		var al = AnnotationWorkList.of(sr, rstream(e2.getAnnotations()));
		var x = JsonSerializer.create().apply(al).build().getSession();
		check(Charset.defaultCharset().name(), x.getFileCharset().name());
		check(Charset.defaultCharset().name(), x.getStreamCharset().name());
	}

	@SerializerConfig(
		initialDepth="$X{abc}",
		maxDepth="$X{xyz}",
		maxIndent="$X{invalid}"
	)
	static class F {}
	static ClassInfo f = ClassInfo.of(F.class);

	@Test void d03_invalidInteger_throwsException() {
		var al = AnnotationWorkList.of(sr, rstream(f.getAnnotations()));
		assertThrows(ConfigException.class, () -> {
			JsonSerializer.create().apply(al).build();
		});
	}
}