/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.myfaces.orchestra.annotation;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.myfaces.orchestra.conversation.annotations.ConversationName;
import org.apache.myfaces.orchestra.conversation.annotations.ConversationRequire;
import org.apache.myfaces.orchestra.viewController.annotations.InitView;
import org.apache.myfaces.orchestra.viewController.annotations.PreProcess;
import org.apache.myfaces.orchestra.viewController.annotations.PreRenderView;
import org.apache.myfaces.orchestra.viewController.annotations.ViewController;

/**
 * Inspects a class for Orchestra annotations, and if found then caches this information
 * for later access.
 * <p>
 * The processing of Class objects is expected to happen only at application startup.
 * <p>
 * Note that annotation scanning is driven by the dependency-injection framework, i.e.
 * only classes declared to the framework are scanned. 
 */
public class AnnotationInfoManager
{
    private final Log log = LogFactory.getLog(AnnotationInfoManager.class);

    private Map<String, AnnotationInfo> annotationsInfoByName = new HashMap<String, AnnotationInfo>();
    private Map<String, AnnotationInfo> annotationsInfoByViewId = new HashMap<String, AnnotationInfo>();

    protected void addAnnotationsInfo(AnnotationInfo annotationInfo)
    {
        if (annotationsInfoByName.containsKey(annotationInfo.getBeanName()))
        {
            log.info("duplicate bean definition: " + annotationInfo.getBeanName());
        }

        annotationsInfoByName.put(annotationInfo.getBeanName(), annotationInfo);

        ViewController viewController = annotationInfo.getViewController();
        if (viewController != null)
        {
            String[] viewIds = viewController.viewIds();
            for (int i = 0; i<viewIds.length; i++)
            {
                String viewId = viewIds[i];

                if (annotationsInfoByViewId.containsKey(annotationInfo.getBeanName()))
                {
                    log.info("duplicate viewId definition: " + annotationInfo.getBeanName());
                }

                annotationsInfoByViewId.put(viewId, annotationInfo);
            }
        }
    }

    public AnnotationInfo getAnnotationInfoByBeanName(String beanName)
    {
        return annotationsInfoByName.get(beanName);
    }

    public AnnotationInfo getAnnotationInfoByViewId(String viewId)
    {
        return annotationsInfoByViewId.get(viewId);
    }

    /**
     * Inspect the provided class for annotations, and if found then cache the info
     * keyed by the specified beanName.
     * <p>
     * Currently the class-level annotations looked for are:
     * <ul>
     * <li>ConversationName
     * <li>ConversationRequire
     * <li>ViewController
     * </ul>
     * <p>
     * If the ViewController annotation is present, then the class is also scanned
     * for related annotations on class methods.
     */
    public void processBeanAnnotations(String beanName, Class<?> clazz)
    {
        ConversationName conversationName = (ConversationName) clazz.getAnnotation(ConversationName.class);
        ViewController viewController = (ViewController) clazz.getAnnotation(ViewController.class);
        ConversationRequire conversationRequire = (ConversationRequire) clazz.getAnnotation(ConversationRequire.class);

        if (conversationName == null && viewController == null && conversationRequire == null)
        {
            return;
        }

        AnnotationInfo annotationInfo = new AnnotationInfo(beanName, clazz);
        annotationInfo.setConversationName(conversationName);
        annotationInfo.setConversationRequire(conversationRequire);
        
        if (viewController != null)
        {
            annotationInfo.setViewController(viewController);
            Method[] methods = clazz.getMethods();
            for (int i = 0; i<methods.length; i++)
            {
                Method method = methods[i];
                if (method.isAnnotationPresent(InitView.class))
                {
                    annotationInfo.setInitViewMethod(method);
                }
                if (method.isAnnotationPresent(PreProcess.class))
                {
                    annotationInfo.setPreProcessMethod(method);
                }
                if (method.isAnnotationPresent(PreRenderView.class))
                {
                    annotationInfo.setPreRenderViewMethod(method);
                }
            }
        }

        addAnnotationsInfo(annotationInfo);
    }
}
