"use strict";
/*! Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const mocha_1 = require("mocha");
const chai_1 = require("chai");
const mona_dish_1 = require("mona-dish");
(0, mocha_1.describe)('early stream tests', () => {
    beforeEach(function () {
        this.probe = [1, 2, 3, 4, 5];
    });
    it("must iterate normal", function () {
        let stream = mona_dish_1.Stream.of(...this.probe);
        let sum = 0;
        stream.each((data) => {
            sum = sum + data;
        });
        (0, chai_1.expect)(sum).to.eq(15);
        let stream2 = mona_dish_1.LazyStream.of(...this.probe);
        sum = 0;
        stream2.each((data) => {
            sum = sum + data;
        });
        (0, chai_1.expect)(sum).to.eq(15);
    });
    it("must iterate filtered", function () {
        let stream = mona_dish_1.Stream.of(...this.probe);
        let sum = 0;
        stream.filter((data) => data != 5).each((data) => {
            sum = sum + data;
        });
        (0, chai_1.expect)(sum).to.eq(10);
        let stream2 = mona_dish_1.LazyStream.of(...this.probe);
        sum = 0;
        stream2.filter((data) => data != 5).each((data) => {
            sum = sum + data;
        });
        (0, chai_1.expect)(sum).to.eq(10);
    });
    it("must onElem", function () {
        let stream = mona_dish_1.Stream.of(...this.probe);
        let sum = 0;
        let sum2 = stream.filter((data) => data != 5).onElem((data) => {
            sum = sum + data;
        }).reduce((el1, el2) => el1 + el2).value;
        (0, chai_1.expect)(sum).to.eq(10);
        (0, chai_1.expect)(sum2).to.eq(10);
        let stream2 = mona_dish_1.LazyStream.of(...this.probe);
        sum = 0;
        sum2 = stream2.filter((data) => data != 5).onElem((data) => {
            sum = sum + data;
        }).reduce((el1, el2) => el1 + el2).value;
        (0, chai_1.expect)(sum).to.eq(10);
        (0, chai_1.expect)(sum2).to.eq(10);
    });
    it("must have a correct first last", function () {
        let first = mona_dish_1.Stream.of(...this.probe).filter((data) => data != 5).onElem(() => {
        }).first().value;
        let last = mona_dish_1.Stream.of(...this.probe).filter((data) => data != 5).onElem(() => {
        }).last().value;
        (0, chai_1.expect)(first).to.eq(1);
        (0, chai_1.expect)(last).to.eq(4);
    });
    it("must have a correct first last lazy", function () {
        let first = mona_dish_1.LazyStream.of(...this.probe).filter((data) => data != 5).onElem((data) => {
            data;
        }).first().value;
        let last = mona_dish_1.Stream.of(...this.probe).filter((data) => data != 5).onElem((data) => {
            data;
        }).last().value;
        (0, chai_1.expect)(first).to.eq(1);
        (0, chai_1.expect)(last).to.eq(4);
    });
    it("must have a correct limits", function () {
        let cnt = 0;
        let last = mona_dish_1.Stream.of(...this.probe).filter((data) => data != 5).limits(2).onElem(() => {
            cnt++;
        }).last().value;
        (0, chai_1.expect)(last).to.eq(2);
        (0, chai_1.expect)(cnt).to.eq(2);
    });
    it("must initialize correctly from assoc array", function () {
        let probe = {
            key1: "val1",
            key2: 2,
            key3: "val3"
        };
        let arr1 = [];
        let arr2 = [];
        mona_dish_1.Stream.ofAssoc(probe).each(item => {
            (0, chai_1.expect)(item.length).to.eq(2);
            arr1.push(item[0]);
            arr2.push(item[1]);
        });
        (0, chai_1.expect)(arr1.join(",")).to.eq("key1,key2,key3");
        (0, chai_1.expect)(arr2.join(",")).to.eq("val1,2,val3");
    });
    it("must have a correct lazy limits", function () {
        let last = mona_dish_1.LazyStream.of(...this.probe).filter((data) => data != 5).limits(2).onElem((data) => {
            data;
        }).last().value;
        (0, chai_1.expect)(last).to.eq(2);
    });
    it("must correctly lazily flatmap", function () {
        let resultingArr = mona_dish_1.LazyStream.of(...this.probe).flatMap((data) => mona_dish_1.LazyStream.of(...[data, 2])).value;
        (0, chai_1.expect)(resultingArr.length == 10).to.be.true;
        (0, chai_1.expect)(resultingArr.join(",")).to.eq("1,2,2,2,3,2,4,2,5,2");
    });
    it("must correctly lazily flatmap with arrays", function () {
        let resultingArr = mona_dish_1.LazyStream.of(...this.probe).flatMap((data) => [data, 2]).value;
        (0, chai_1.expect)(resultingArr.length == 10).to.be.true;
        (0, chai_1.expect)(resultingArr.join(",")).to.eq("1,2,2,2,3,2,4,2,5,2");
    });
    it("must correctly early flatmap", function () {
        let resultingArr = mona_dish_1.Stream.of(...this.probe).flatMap((data) => mona_dish_1.Stream.of(...[data, 2])).value;
        (0, chai_1.expect)(resultingArr.length == 10).to.be.true;
        (0, chai_1.expect)(resultingArr.join(",")).to.eq("1,2,2,2,3,2,4,2,5,2");
    });
    it("must correctly early flatmap with arrays", function () {
        let resultingArr = mona_dish_1.Stream.of(...this.probe).flatMap((data) => [data, 2]).value;
        (0, chai_1.expect)(resultingArr.length == 10).to.be.true;
        (0, chai_1.expect)(resultingArr.join(",")).to.eq("1,2,2,2,3,2,4,2,5,2");
    });
    it("must correctly flatmap intermixed", function () {
        let resultingArr = mona_dish_1.LazyStream.of(...this.probe).flatMap((data) => mona_dish_1.Stream.of(...[data, 2])).value;
        (0, chai_1.expect)(resultingArr.length == 10).to.be.true;
        (0, chai_1.expect)(resultingArr.join(",")).to.eq("1,2,2,2,3,2,4,2,5,2");
    });
    it("must correctly flatmap intermixed2", function () {
        let resultingArr = mona_dish_1.Stream.of(...this.probe).flatMap((data) => mona_dish_1.LazyStream.of(...[data, 2])).value;
        (0, chai_1.expect)(resultingArr.length == 10).to.be.true;
        (0, chai_1.expect)(resultingArr.join(",")).to.eq("1,2,2,2,3,2,4,2,5,2");
    });
    it("must correctly pass anyMatch allMatch noneMatch", function () {
        let anyMatch = mona_dish_1.LazyStream.of(...this.probe).anyMatch((item) => item == 3);
        let allMatch = mona_dish_1.LazyStream.of(...this.probe).allMatch((item) => item < 6);
        let noneMatch = mona_dish_1.LazyStream.of(...this.probe).noneMatch((item) => item > 5);
        (0, chai_1.expect)(anyMatch).to.be.true;
        (0, chai_1.expect)(allMatch).to.be.true;
        (0, chai_1.expect)(noneMatch).to.be.true;
    });
    it("must correctly pass anyMatch allMatch noneMatch early", function () {
        let anyMatch = mona_dish_1.Stream.of(...this.probe).anyMatch((item) => item == 3);
        let allMatch = mona_dish_1.Stream.of(...this.probe).allMatch((item) => item < 6);
        let noneMatch = mona_dish_1.Stream.of(...this.probe).noneMatch((item) => item > 5);
        (0, chai_1.expect)(anyMatch).to.be.true;
        (0, chai_1.expect)(allMatch).to.be.true;
        (0, chai_1.expect)(noneMatch).to.be.true;
    });
    it("must sort correctly", function () {
        let probe = [1, 5, 3, 2, 4];
        let res = mona_dish_1.Stream.of(...probe)
            .sort((el1, el2) => el1 - el2)
            .collect(new mona_dish_1.ArrayCollector());
        (0, chai_1.expect)(res.join(",")).to.eq("1,2,3,4,5");
    });
    it("must sort correctly lazy", function () {
        let probe = [1, 5, 3, 2, 4];
        let res = mona_dish_1.LazyStream.of(...probe)
            .sort((el1, el2) => el1 - el2)
            .collect(new mona_dish_1.ArrayCollector());
        (0, chai_1.expect)(res.join(",")).to.eq("1,2,3,4,5");
    });
});
//# sourceMappingURL=StreamTest.spec.js.map