/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.record;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Constructor;
import java.nio.ByteBuffer;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.record.ByteBufferInputStream;
import org.apache.kafka.common.record.ByteBufferOutputStream;
import org.apache.kafka.common.record.CompressionType;
import org.apache.kafka.common.record.Record;
import org.apache.kafka.common.utils.Utils;

public class Compressor {
    private static final float COMPRESSION_RATE_DAMPING_FACTOR = 0.9f;
    private static final float COMPRESSION_RATE_ESTIMATION_FACTOR = 1.05f;
    private static final int COMPRESSION_DEFAULT_BUFFER_SIZE = 1024;
    private static final float[] TYPE_TO_RATE;
    private static MemoizingConstructorSupplier snappyOutputStreamSupplier;
    private static MemoizingConstructorSupplier lz4OutputStreamSupplier;
    private static MemoizingConstructorSupplier snappyInputStreamSupplier;
    private static MemoizingConstructorSupplier lz4InputStreamSupplier;
    private final CompressionType type;
    private final DataOutputStream appendStream;
    private final ByteBufferOutputStream bufferStream;
    private final int initPos;
    public long writtenUncompressed;
    public long numRecords;
    public float compressionRate;
    public long maxTimestamp;

    public Compressor(ByteBuffer buffer, CompressionType type) {
        this.type = type;
        this.initPos = buffer.position();
        this.numRecords = 0L;
        this.writtenUncompressed = 0L;
        this.compressionRate = 1.0f;
        this.maxTimestamp = -1L;
        if (type != CompressionType.NONE) {
            buffer.position(this.initPos + 12 + 22);
        }
        this.bufferStream = new ByteBufferOutputStream(buffer);
        this.appendStream = Compressor.wrapForOutput(this.bufferStream, type, 1024);
    }

    public ByteBuffer buffer() {
        return this.bufferStream.buffer();
    }

    public double compressionRate() {
        return this.compressionRate;
    }

    public void close() {
        try {
            this.appendStream.close();
        }
        catch (IOException e) {
            throw new KafkaException(e);
        }
        if (this.type != CompressionType.NONE) {
            ByteBuffer buffer = this.bufferStream.buffer();
            int pos = buffer.position();
            buffer.position(this.initPos);
            buffer.putLong(this.numRecords - 1L);
            buffer.putInt(pos - this.initPos - 12);
            Record.write(buffer, this.maxTimestamp, null, null, this.type, 0, -1);
            int valueSize = pos - this.initPos - 12 - 22;
            buffer.putInt(this.initPos + 12 + 18, valueSize);
            long crc = Record.computeChecksum(buffer, this.initPos + 12 + 4, pos - this.initPos - 12 - 4);
            Utils.writeUnsignedInt(buffer, this.initPos + 12 + 0, crc);
            buffer.position(pos);
            this.compressionRate = (float)buffer.position() / (float)this.writtenUncompressed;
            Compressor.TYPE_TO_RATE[this.type.id] = TYPE_TO_RATE[this.type.id] * 0.9f + this.compressionRate * 0.100000024f;
        }
    }

    public void putLong(long value) {
        try {
            this.appendStream.writeLong(value);
        }
        catch (IOException e) {
            throw new KafkaException("I/O exception when writing to the append stream, closing", e);
        }
    }

    public void putInt(int value) {
        try {
            this.appendStream.writeInt(value);
        }
        catch (IOException e) {
            throw new KafkaException("I/O exception when writing to the append stream, closing", e);
        }
    }

    public void put(ByteBuffer buffer) {
        try {
            this.appendStream.write(buffer.array(), buffer.arrayOffset(), buffer.limit());
        }
        catch (IOException e) {
            throw new KafkaException("I/O exception when writing to the append stream, closing", e);
        }
    }

    public void putByte(byte value) {
        try {
            this.appendStream.write(value);
        }
        catch (IOException e) {
            throw new KafkaException("I/O exception when writing to the append stream, closing", e);
        }
    }

    public void put(byte[] bytes, int offset, int len) {
        try {
            this.appendStream.write(bytes, offset, len);
        }
        catch (IOException e) {
            throw new KafkaException("I/O exception when writing to the append stream, closing", e);
        }
    }

    public long putRecord(long timestamp, byte[] key, byte[] value, CompressionType type, int valueOffset, int valueSize) {
        long crc = Record.computeChecksum(timestamp, key, value, type, valueOffset, valueSize);
        byte attributes = Record.computeAttributes(type);
        this.putRecord(crc, attributes, timestamp, key, value, valueOffset, valueSize);
        return crc;
    }

    public long putRecord(long timestamp, byte[] key, byte[] value) {
        return this.putRecord(timestamp, key, value, CompressionType.NONE, 0, -1);
    }

    private void putRecord(long crc, byte attributes, long timestamp, byte[] key, byte[] value, int valueOffset, int valueSize) {
        this.maxTimestamp = Math.max(this.maxTimestamp, timestamp);
        Record.write(this, crc, attributes, timestamp, key, value, valueOffset, valueSize);
    }

    public void recordWritten(int size) {
        ++this.numRecords;
        this.writtenUncompressed += (long)size;
    }

    public long numRecordsWritten() {
        return this.numRecords;
    }

    public long estimatedBytesWritten() {
        if (this.type == CompressionType.NONE) {
            return this.bufferStream.buffer().position();
        }
        return (long)((float)this.writtenUncompressed * TYPE_TO_RATE[this.type.id] * 1.05f);
    }

    public static DataOutputStream wrapForOutput(ByteBufferOutputStream buffer, CompressionType type, int bufferSize) {
        try {
            switch (type) {
                case NONE: {
                    return new DataOutputStream(buffer);
                }
                case GZIP: {
                    return new DataOutputStream(new GZIPOutputStream((OutputStream)buffer, bufferSize));
                }
                case SNAPPY: {
                    try {
                        OutputStream stream = (OutputStream)snappyOutputStreamSupplier.get().newInstance(buffer, bufferSize);
                        return new DataOutputStream(stream);
                    }
                    catch (Exception e) {
                        throw new KafkaException(e);
                    }
                }
                case LZ4: {
                    try {
                        OutputStream stream = (OutputStream)lz4OutputStreamSupplier.get().newInstance(buffer);
                        return new DataOutputStream(stream);
                    }
                    catch (Exception e) {
                        throw new KafkaException(e);
                    }
                }
            }
            throw new IllegalArgumentException("Unknown compression type: " + (Object)((Object)type));
        }
        catch (IOException e) {
            throw new KafkaException(e);
        }
    }

    public static DataInputStream wrapForInput(ByteBufferInputStream buffer, CompressionType type, byte messageVersion) {
        try {
            switch (type) {
                case NONE: {
                    return new DataInputStream(buffer);
                }
                case GZIP: {
                    return new DataInputStream(new GZIPInputStream(buffer));
                }
                case SNAPPY: {
                    try {
                        InputStream stream = (InputStream)snappyInputStreamSupplier.get().newInstance(buffer);
                        return new DataInputStream(stream);
                    }
                    catch (Exception e) {
                        throw new KafkaException(e);
                    }
                }
                case LZ4: {
                    try {
                        InputStream stream = (InputStream)lz4InputStreamSupplier.get().newInstance(buffer, messageVersion == 0);
                        return new DataInputStream(stream);
                    }
                    catch (Exception e) {
                        throw new KafkaException(e);
                    }
                }
            }
            throw new IllegalArgumentException("Unknown compression type: " + (Object)((Object)type));
        }
        catch (IOException e) {
            throw new KafkaException(e);
        }
    }

    static {
        int maxTypeId = -1;
        for (CompressionType type : CompressionType.values()) {
            maxTypeId = Math.max(maxTypeId, type.id);
        }
        TYPE_TO_RATE = new float[maxTypeId + 1];
        for (CompressionType type : CompressionType.values()) {
            Compressor.TYPE_TO_RATE[type.id] = type.rate;
        }
        snappyOutputStreamSupplier = new MemoizingConstructorSupplier(new ConstructorSupplier(){

            @Override
            public Constructor get() throws ClassNotFoundException, NoSuchMethodException {
                return Class.forName("org.xerial.snappy.SnappyOutputStream").getConstructor(OutputStream.class, Integer.TYPE);
            }
        });
        lz4OutputStreamSupplier = new MemoizingConstructorSupplier(new ConstructorSupplier(){

            @Override
            public Constructor get() throws ClassNotFoundException, NoSuchMethodException {
                return Class.forName("org.apache.kafka.common.record.KafkaLZ4BlockOutputStream").getConstructor(OutputStream.class);
            }
        });
        snappyInputStreamSupplier = new MemoizingConstructorSupplier(new ConstructorSupplier(){

            @Override
            public Constructor get() throws ClassNotFoundException, NoSuchMethodException {
                return Class.forName("org.xerial.snappy.SnappyInputStream").getConstructor(InputStream.class);
            }
        });
        lz4InputStreamSupplier = new MemoizingConstructorSupplier(new ConstructorSupplier(){

            @Override
            public Constructor get() throws ClassNotFoundException, NoSuchMethodException {
                return Class.forName("org.apache.kafka.common.record.KafkaLZ4BlockInputStream").getConstructor(InputStream.class, Boolean.TYPE);
            }
        });
    }

    private static class MemoizingConstructorSupplier {
        final ConstructorSupplier delegate;
        volatile transient boolean initialized;
        transient Constructor value;

        public MemoizingConstructorSupplier(ConstructorSupplier delegate) {
            this.delegate = delegate;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public Constructor get() throws NoSuchMethodException, ClassNotFoundException {
            if (!this.initialized) {
                MemoizingConstructorSupplier memoizingConstructorSupplier = this;
                synchronized (memoizingConstructorSupplier) {
                    if (!this.initialized) {
                        Constructor constructor;
                        this.value = constructor = this.delegate.get();
                        this.initialized = true;
                        return constructor;
                    }
                }
            }
            return this.value;
        }
    }

    private static interface ConstructorSupplier {
        public Constructor get() throws ClassNotFoundException, NoSuchMethodException;
    }
}

