##
# DG-505 Elan Orion Electrical System
#	by Benedikt Wolf (D-ECHO) 08/2021

#	adapted from: Cessna 172p Electrical System


var volts = 0.0;

var bus_volts = 0.0;

var ammeter_ave = 0.0;

# Initialize properties
var electrical	=	props.globals.initNode("/systems/electrical");
var serviceable	=	electrical.initNode("serviceable", 1, "BOOL");
var batt_prop	=	electrical.initNode("battery");
var output_prop	=	electrical.initNode("outputs");
var cb_prop	=	props.globals.initNode("/controls/circuit-breakers");

var volts	=	electrical.initNode("volts", 0.0, "DOUBLE");
var amps	=	electrical.initNode("amps",  0.0, "DOUBLE");
var master_sw	=	props.globals.initNode("/controls/electric/battery-selector-switch", 0, "INT");	# three-way switch: -1 = front batt, 0 = off, 1 = rear batt

var radio	=	output_prop.initNode("comm[0]", 0.0, "DOUBLE");
var flarm	=	output_prop.initNode("flarm", 0.0, "DOUBLE");
var s3		=	output_prop.initNode("S3", 0.0, "DOUBLE");
var lx		=	output_prop.initNode("lx", 0.0, "DOUBLE");
var fin_ind_empty =	output_prop.initNode("fin-tank-b-indicator-empty", 0.0, "DOUBLE");
var fin_ind_full  =	output_prop.initNode("fin-tank-b-indicator-full",  0.0, "DOUBLE");

var comm_ptt	=	props.globals.getNode("/instrumentation/comm[0]/ptt");

var fin_tank_b_lb =	props.globals.getNode("/payload/weight[7]/weight-lb");

var battery_front	=	props.globals.getNode("/dg505/electrical/battery-front", 1);

# Array of circuit breakers
var circuit_breakers = {
	master: cb_prop.initNode("master", 1, "BOOL"),
};

var freeze_replay	=	props.globals.getNode("/sim/freeze/replay-state");

##
# Battery model class.
#

var BatteryClass = {
	# Constructor
	new: func( ideal_volts, ideal_amps, amp_hours, charge_amps, n ){
		var charge_prop	= batt_prop.getNode( "charge["~n~"]" );
		var charge	= nil;
		if( getprop("/systems/electrical/battery/charge["~n~"]") != nil ){			# If the battery charge has been set from a previous FG instance
			charge = charge_prop.getDoubleValue();
		} else {
			charge = 1.0;
			charge_prop = batt_prop.initNode("charge["~n~"]", 1.0, "DOUBLE");
		}
		var obj = {
			parents: [BatteryClass],
			ideal_volts: ideal_volts,
			ideal_amps: ideal_amps,
			amp_hours: amp_hours,
			charge_amps: charge_amps,
			charge: charge,
			charge_prop: charge_prop,
			n: n 
		};
		return obj;
	},
	# Passing in positive amps means the battery will be discharged.
	# Negative amps indicates a battery charge.
	apply_load: func( amps, dt ){
		var old_charge = me.charge_prop.getDoubleValue();
		if( freeze_replay.getBoolValue() ){
			return me.amp_hours * old_charge;
		}
		var amphrs_used = amps * dt / 3600.0;
		var fraction_used = amphrs_used / me.amp_hours;
		
		var new_charge = std.max(0.0, std.min(old_charge - fraction_used, 1.0));
		
		if (new_charge < 0.1 and old_charge_percent >= 0.1)
			gui.popupTip("Warning: Low battery! Enable alternator or apply external power to recharge battery!", 10);
		me.charge = new_charge;
		me.charge_prop.setDoubleValue( new_charge );
		return me.amp_hours * new_charge;
	},
	# Return output volts based on percent charged.  Currently based on a simple
	# polynomial percent charge vs. volts function.
	get_output_volts: func() {
		var x = 1.0 - me.charge;
		var tmp = -(3.0 * x - 1.0);
		var factor = ( math.pow( tmp, 5) + 32 ) / 32;
		return me.ideal_volts * factor;
	},
	# Return output amps available.  This function is totally wrong and should be
	# fixed at some point with a more sensible function based on charge percent.
	# There is probably some physical limits to the number of instantaneous amps
	# a battery can produce (cold cranking amps?)
	get_output_amps: func() {
		var x = 1.0 - me.charge;
		var tmp = -(3.0 * x - 1.0);
		var factor = ( math.pow( tmp, 5) + 32) / 32;
		return me.ideal_amps * factor;
	},
	# Set the current charge instantly to 100 %.
	reset_to_full_charge: func() {
		me.apply_load(-(1.0 - me.charge) * me.amp_hours, 3600);
	},
	# Get current charge
	get_charge: func() {
		return me.charge;
	}
};

############################
####	Battery Packs	####
############################

##	example glider battery: https://shop.segelflugbedarf24.de/Flugzeugausstattung/Akkus-Energieversorgung/Sonnenschein-Dryfit-A512-12V/6-5-Ah::731.html
##				http://www.sonnenschein.org/A500.htm	(A512-6.5S)
##				ideal volts: 12.0
##				ideal amps: 0.325 (max. 80 / 300 for 5 sec))
##				amp hours: 6.5
##				charge amps: 25

##	DG505 Elan Orion glider battery (ref. 1, p. 6.3 ):
##		ideal volts: 12
##		amp hours: 10

var	front_batt = BatteryClass.new( 12, 0.325, 10, 25, 0);
var	rear_batt = BatteryClass.new( 12, 0.325, 10, 25, 1);

var reset_battery = func {
	front_batt.reset_to_full_charge();
	rear_batt.reset_to_full_charge();
}
var reset_circuit_breakers = func {
	# Reset circuit breakers
	foreach(var cb; keys(circuit_breakers)){
		circuit_breakers[cb].setBoolValue( 1 );
	}
}

##
# This is the main electrical system update function.
#

var ElectricalSystemUpdater = {
	new : func {
		var m = {
			parents: [ElectricalSystemUpdater]
		};
		# Request that the update function be called each frame
		m.loop = updateloop.UpdateLoop.new(components: [m], update_period: 0.0, enable: 0);
		return m;
	},
	
	enable: func {
		me.loop.reset();
		me.loop.enable();
	},
	
	disable: func {
		me.loop.disable();
	},
	
	reset: func {
		# Do nothing
	},
	
	update: func (dt) {
		update_bus(dt);
	}
};


var update_bus = func (dt) {
	var load = 0.0;
	var bus_volts = 0.0;
	
	var master_pos = master_sw.getIntValue();
	
	if( circuit_breakers.master.getBoolValue() and master_pos != 0){
		if( master_pos == -1 and battery_front.getBoolValue() ){
			bus_volts = front_batt.get_output_volts();
		} else if ( master_pos == 1 ){
			bus_volts = rear_batt.get_output_volts();
		}
	}
	if( !serviceable.getBoolValue() ){
		bus_volts = 0.0;
	}
	
	# switch state
	load += cockpit_bus( bus_volts );
	
	if ( load > 300 ) {
		circuit_breakers.master.setBoolValue( 0 );
	}
	
	if( master_pos == -1 and battery_front.getBoolValue() ){
		front_batt.apply_load( load, dt );
	} else if ( master_pos == 1 ){
		rear_batt.apply_load( load, dt );
	}
}

#Load sources:
#	com:		https://www.becker-avionics.com/wp-content/uploads/00_Literature/62Series_DS_en.pdf, p.2
#	vario:		http://www.lx-avionik.de/produkte/s3/
#	flarm:		http://flarm.com/wp-content/uploads/man/FLARM_InstallationManual_D.pdf
#	flarm display:	https://www.air-store.eu/Display-V3-FLARM
#	lx8080:		http://www.lx-avionik.de/wp/download/manuals/lx90xx-80xxInstallationManualEnglish_rev23.pdf

var cockpit_bus = func( bv ) {
	var load = 0.0;
	var bus_volts = bv;
	
	# Electrical Consumers (Instruments)
	
	if( bus_volts > 9 ){	# minimal working voltage
		# Radio
		radio.setDoubleValue( bus_volts );
		if( comm_ptt.getBoolValue() ){
			load += 18 / bus_volts; # transmitting: <= 2 A at 12V
		}else{
			load += 1.56 / bus_volts; # standby: <= 140 mA at 12V
		}
	
		# Vario
		s3.setDoubleValue( bus_volts );
		load += 1.380 / bus_volts; # 95mA - 135mA depending on display brightness setting at 12V
		
	} else {
		radio.setDoubleValue( 0.0 );
		s3.setDoubleValue( 0.0 );
	}
	
	# FLARM
	if( bus_volts > 8 ){	# minimal working voltage
		flarm.setDoubleValue( bus_volts );
		load += 0.66 / bus_volts; #FLARM
		load += 0.12 / bus_volts; #FLARM display
	} else {
		flarm.setDoubleValue( 0.0 );
	}
	
	# LX8080
	if( bus_volts >= 10 ){	# minimal working voltage (p.9)
		lx.setDoubleValue( bus_volts );
		load += 3 / bus_volts; # 250mA at 12V, max. brightness (p.7)
	} else {
		lx.setDoubleValue( 0.0 );
	}
	
	# ref. 1, p. 7.5: Fin Tank B Indication Lights
	if( fin_tank_b_lb.getDoubleValue() < 0.01 ){
		fin_ind_empty.setDoubleValue( bus_volts );
		fin_ind_full.setDoubleValue( 0.0 );
	} else {
		fin_ind_full.setDoubleValue( bus_volts );
		fin_ind_empty.setDoubleValue( 0.0 );
	}
	
	amps.setDoubleValue( load );
	volts.setDoubleValue( bus_volts );
	
	return load;
}


##
# Initialize the electrical system
#

var system_updater = ElectricalSystemUpdater.new();

setlistener("/sim/signals/fdm-initialized", func {
	reset_circuit_breakers();
	
	system_updater.enable();
	print("Electrical System initialized");
});

