# Simulation of Geiger Engineering ADI (Advanced Drive Interface) started 07/2022 by Benedikt Wolf based on

# A3XX Lower ECAM Canvas
# Joshua Davidson (Octal450)

# References:
#	https://www.geigerengineering.de/_Resources/Persistent/5081e479d20ad2b8fa95bcb2088f75b275ee6190/Handbuch_Manual_E_Drive%20komplett_V1_999_DE_EN.pdf

var ADI_main = nil;
var ADI_display = nil;

var base		=	props.globals.getNode("/instrumentation/adi/");

var engine_properties = {
	power:		props.globals.getNode("/fdm/jsbsim/propulsion/engine[0]/power-hp", 1),
	rpm:		props.globals.getNode("/engines/engine[0]/rpm", 1),
	throttle:	props.globals.getNode("/fdm/jsbsim/fcs/adi/throttle-system-raw", 1),
};

var accel = props.globals.getNode("/accelerations/pilot-g", 1);

var groundspeed = props.globals.getNode("/velocities/groundspeed-kt", 1);

var vario = props.globals.getNode("/instrumentation/vertical-speed-indicator/indicated-speed-mps", 1);
var alt_baro = props.globals.getNode("/instrumentation/altimeter/indicated-altitude-ft", 1);

var time = props.globals.getNode("/instrumentation/clock/indicated-short-string", 1);

var volts = props.globals.getNode("systems/electrical/outputs/adi", 1);

var batt_temp	= props.globals.getNode("/systems/electrical/battery[0]/temperature-degc", 1);
var motor_temp	= props.globals.initNode("/engines/engine[0]/engine-temp-degc", 15.0, "DOUBLE");
var ctrl_temp	= props.globals.initNode("/engines/engine[0]/mc300-temp-degc", 15.0, "DOUBLE");

var instrument_dir = "Aircraft/Swan/Models/Instruments/ADI/";

var elapsed_sec = props.globals.getNode("/sim/time/elapsed-sec", 1);

var power_controller_release = props.globals.initNode("/fdm/jsbsim/fcs/mc300/enable-throttle", 0, "BOOL");

var error_audio_signal = base.initNode("error-audio-signal", 0, "BOOL");

var hobbs_min	= props.globals.initNode("/engines/engine[0]/hobbs-min", 0.0, "DOUBLE");

var error = 0;		#	0 = no error
			#	>0 errors

var canvas_ADI_base = {
	init: func(canvas_group, file) {
		var font_mapper = func(family, weight) {
			if( find( "monospace", family ) != -1 ){
				return "monoMMM_5.ttf";
			}elsif( find( "Liberation Mono", family ) != -1 ){
				if (weight == "bold"){
					return "LiberationFonts/LiberationMono-Bold.ttf";
				}elsif(weight == "normal"){
					return "LiberationFonts/LiberationMono-Regular.ttf";
				}
			}elsif( family == "'Liberation Sans'" ){
				if (weight == "bold"){
					return "LiberationFonts/LiberationSans-Bold.ttf";
				}elsif(weight == "normal"){
					return "LiberationFonts/LiberationSans-Regular.ttf";
				}
			}elsif( family == "'Liberation Serif'" ){
				if (weight == "bold"){
					return "LiberationFonts/LiberationSerif-Bold.ttf";
				}elsif(weight == "normal"){
					return "LiberationFonts/LiberationSerif-Regular.ttf";
				}
			}elsif( family == "Orbitron" ){
				if( weight == "bold" ){
					return "Orbitron/Orbitron-Bold.ttf";
				} else {
					return "Orbitron/Orbitron-Regular.ttf";
				}
			} elsif( family == "'Orbitron Medium'" ){
				return "Orbitron/Orbitron-Medium.ttf";
			} else {
				print( "No matching Font found: " );
				debug.dump( caller(1) );
				print( family );
				print( weight );
				return "LiberationFonts/LiberationSans-Regular.ttf";
			}
		};

		
		canvas.parsesvg(canvas_group, file, {'font-mapper': font_mapper});

		var svg_keys = me.getKeys();
		
		foreach (var key; svg_keys) {
			me[key] = canvas_group.getElementById(key);
			var clip_el = canvas_group.getElementById(key ~ "_clip");
			if (clip_el != nil) {
				clip_el.setVisible(0);
				var tran_rect = clip_el.getTransformedBounds();
				var clip_rect = sprintf("rect(%d,%d, %d,%d)", 
				tran_rect[1], # 0 ys
				tran_rect[2], # 1 xe
				tran_rect[3], # 2 ye
				tran_rect[0]); #3 xs
				#   coordinates are top,right,bottom,left (ys, xe, ye, xs) ref: l621 of simgear/canvas/CanvasElement.cxx
				me[key].set("clip", clip_rect);
				me[key].set("clip-frame", canvas.Element.PARENT);
			}
		}

		me.page = canvas_group;

		return me;
	},
	getKeys: func() {
		return [];
	},
	update: func() {	
		if ( volts.getDoubleValue() > 4) {
			ADI_main.page.show();
			ADI_main.update();
		} else {
			ADI_main.page.hide();
		}
	},
};

var rpm = 0.0;
var power = 0.0;

	
	
var canvas_ADI_main = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_ADI_main , canvas_ADI_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return ["airspeed.needle", "g_accel.needle", "vert_speed.needle", "rpm.needle", "rpm.digits", "throttle.digits", "pwr.needle", "pwr.digits", "time.digits", "alt.digits", "batt_pct.digits", "battery_charge_group", "battery_charge.green", "battery_charge.amber", "battery_charge.red", "mot_temp.digits", "mc_temp.digits", "bat_temp.digits","error_message","error_message.text"];
	},
	update: func() {
		me["time.digits"].setText( time.getValue() );
		
		me["airspeed.needle"].setRotation( math.min( groundspeed.getDoubleValue() * 1.852, 200 ) * 1.35 * D2R );
		
		me["g_accel.needle"].setRotation( math.clamp( accel.getDoubleValue(), 0, 3 ) * 20 * D2R );
		
		me["vert_speed.needle"].setRotation( math.clamp( vario.getDoubleValue(), -4, 4 ) * 7.5 * D2R );
		
		me["alt.digits"].setText( sprintf("%4d", math.round( alt_baro.getDoubleValue() * FT2M ) ) );
		
		rpm = engine_properties.rpm.getDoubleValue();
		me["rpm.digits"].setText( sprintf("%2d", math.round( rpm / 100 ) ) );
		me["rpm.needle"].setRotation( math.clamp( rpm, 0, 2500 ) * 0.024 * D2R );
		
		me["throttle.digits"].setText( sprintf("%3d", math.round( engine_properties.throttle.getDoubleValue() * 100 ) ) ~ "%" );
		
		power = engine_properties.power.getDoubleValue() * 0.7457; # hp to kW
		me["pwr.digits"].setText( sprintf("%2d", math.round( power ) ) );
		me["pwr.needle"].setRotation( math.clamp( power, 0, 40 ) * 1.5 * D2R );
		
		var batt_pct = swan.battery.charge_percent;
		me["batt_pct.digits"].setText( sprintf("%1d", math.round( batt_pct * 100 ) ) );
		
		me["battery_charge.green"].setTranslation( 0, math.max( 1 - batt_pct, 0 ) * 325 );
		me["battery_charge.amber"].setTranslation( 0, math.max( 1 - batt_pct - 0.7, 0 ) * 325 );
		me["battery_charge.red"].setTranslation( 0, math.max( 1 - batt_pct - 0.79, 0 ) * 325 );
		
		me["mot_temp.digits"].setText( sprintf("%2d", math.round( motor_temp.getDoubleValue() ) ) );
		me["mc_temp.digits"].setText( sprintf("%2d", math.round( ctrl_temp.getDoubleValue() ) ) );
		me["bat_temp.digits"].setText( sprintf("%2d", math.round( batt_temp.getDoubleValue() ) ) );
		
		if( error == 0 ){
			error_audio_signal.setBoolValue( 0 );
			me["error_message"].hide();
		}else{
			error_audio_signal.setBoolValue( 1 );
			me["error_message"].show();
			if( error == 1 ){
				me["error_message.text"].setText( "Error: U<Umin  " );
			}elsif( error == 2 ){
				me["error_message.text"].setText( "Error: T.Contr." );
			}elsif( error == 3 ){
				me["error_message.text"].setText( "Error: T.Motor " );
			}elsif( error == 4 ){
				me["error_message.text"].setText( "Error: Battery " );
			}elsif( error == 5 ){
				me["error_message.text"].setText( "Error: I_Limit " );
			}elsif( error == 6 ){
				me["error_message.text"].setText( "Error: Cut OFF " );
			}elsif( error == 7 ){
				me["error_message.text"].setText( "Error: external" );
			}elsif( error == 8 ){
				me["error_message.text"].setText( "Error: Poti min" );
			}elsif( error == 9 ){
				me["error_message.text"].setText( "Error: SD-Card " );
			}else{
				screen.log.write( "Advanced Drive Interface: Unknown Error Code, reset to 0 (no error)" );
				error = 0;
			}
		}
	}
	
};

var base_updater = maketimer( 0.02, canvas_ADI_base.update );
base_updater.simulatedTime = 1;

setlistener("sim/signals/fdm-initialized", func {
	ADI_display = canvas.new({
		"name": "ADI",
		"size": [512, 892],
		"view": [512, 892],
		"mipmapping": 1
	});
	ADI_display.addPlacement({"node": "adi.screen"});
	var groupMain = ADI_display.createGroup();


	ADI_main = canvas_ADI_main.new(groupMain, instrument_dir~"adi.svg");

	base_updater.start();
}); 

var ctrl_time = nil;
var ctrl_prev = 0;

var controller_button = func( a ){
	if( a ){
		if( getprop("/fdm/jsbsim/fcs/mc300/throttle-at-zero") ){
			ctrl_time = elapsed_sec.getDoubleValue();
		} else {
			ctrl_time = nil;
			error = 8;
		}
	} elsif( ctrl_prev ) {
		if( ctrl_time != nil and ( elapsed_sec.getDoubleValue() - ctrl_time ) > 1.5 ){
			power_controller_release.setBoolValue( 1 );
		} else {
			power_controller_release.setBoolValue( 0 );
		}
		ctrl_time = nil;
		error = 0;
	}
	ctrl_prev = a;
}
setlistener("/fdm/jsbsim/fcs/mc300/throttle-at-zero", func( i ){
	if( !i.getBoolValue() and ctrl_prev != 0 ){
		ctrl_time = nil;
	}
	if( i.getBoolValue() and error == 8 ){
		error = 0;
	}
});

var last_hobbs = 0.0;

var hobbs_meter = func {
	if( engine_properties.rpm.getDoubleValue() > 10 ){
		hobbs_min.setDoubleValue( hobbs_min.getDoubleValue() + ( elapsed_sec.getDoubleValue() - last_hobbs ) / 60 );
	}
	last_hobbs = elapsed_sec.getDoubleValue();
}

var hobbs_meter_update = maketimer( 10.0, hobbs_meter );
hobbs_meter_update.simulatedTime = 1;
hobbs_meter_update.start();
	
