# Copyright (C) 2004,2005 by SICEm S.L.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

import gettext
import os
import urllib
import urlparse

import gtk
import gobject

from gazpacho import __version__
from gazpacho.actioneditor import GActionsView
from gazpacho.app.bars import bar_manager
from gazpacho.app.dialogs import SingleCloseConfirmationDialog, \
     MultipleCloseConfirmationDialog
from gazpacho.clipboard import clipboard, ClipboardWindow
from gazpacho.constants import MISSING_ICON
from gazpacho.command import CommandStackView
from gazpacho.commandmanager import command_manager
from gazpacho.config import config
from gazpacho.dialogs import open, save, error, messagedialog
from gazpacho.editor import Editor
from gazpacho.environ import environ
from gazpacho.loader.loader import ParseError
from gazpacho.palette import palette
from gazpacho.placeholder import Placeholder
from gazpacho.project import Project
from gazpacho.sizegroupeditor import SizeGroupView
from gazpacho.util import rebuild
from gazpacho.widget import Widget
from gazpacho.widgetview import WidgetTreeView
from gazpacho.widgetregistry import widget_registry
from gazpacho.uimstate import WidgetUIMState, ActionUIMState, SizeGroupUIMState

_ = gettext.gettext

__all__ = ['Application']

class Application(object):

    # DND information
    TARGET_TYPE_URI = 100
    targets = [('text/uri-list', 0, TARGET_TYPE_URI)]
    
    def __init__(self):
        # The WidgetAdaptor that we are about to add to a container. None if no
        # class is to be added. This also has to be in sync with the depressed
        # button in the Palette
        self._add_class = None

        # This is our current project
        self._project = None

        # The uim_states maps notebook page numbers to uim states
        self._uim_states = {}
        # The uim state that is currently active
        self._activ_uim_state = None

        # debugging windows
        self._command_stack_window = None
        self._clipboard_window = None

        # here we put views that should update when changing the project
        self._project_views = []

        self._active_view = None
        self._projects = []
        self._show_structure = False
        
        self._window = self._application_window_create()

    def _application_window_create(self):
        application_window = gtk.Window(gtk.WINDOW_TOPLEVEL)
        application_window.connect('delete-event', self._delete_event_cb)
        application_window.set_size_request(600, -1)
        iconfilename = environ.find_pixmap('gazpacho-icon.png')
        gtk.window_set_default_icon_from_file(iconfilename)

        # Layout them on the window
        main_vbox = gtk.VBox()
        application_window.add(main_vbox)

        # Create actions that are always enabled
        bar_manager.add_actions(
            'Normal',
            # File menu
            ('FileMenu', None, _('_File')),
            ('New', gtk.STOCK_NEW, None, None,
             _('New Project'), self._new_cb),
            ('Open', gtk.STOCK_OPEN, None, None,
             _('Open Project'), self._open_cb),
            ('Quit', gtk.STOCK_QUIT, None, None,
             _('Quit'), self._quit_cb),

            # Edit menu
            ('EditMenu', None, _('_Edit')),
            
            # Object menu
            ('ObjectMenu', None, _('_Objects')),
            
            # Project menu
            ('ProjectMenu', None, _('_Project')),
            # (projects..)
            
            # Debug menu
            ('DebugMenu', None, _('_Debug')),
            ('DumpData', None, _('Dump data'), '<control>M',
              _('Dump debug data'), self._dump_data_cb),
            ('Reload', None, _('Reload'), None,
             _('Reload python code'), self._reload_cb),
            
            # Help menu
            ('HelpMenu', None, _('_Help')),
            ('About', gtk.STOCK_ABOUT, None, None, _('About Gazpacho'),
             self._about_cb),
            )

        # Toggle actions
        bar_manager.add_toggle_actions(
            'Normal',
            ('ShowStructure', None, _('Show _structure'), '<control><shift>t',
             _('Show container structure'), self._show_structure_cb, False),
            ('ShowCommandStack', None, _('Show _command stack'), 'F3',
             _('Show the command stack'), self._show_command_stack_cb, False),
            ('ShowClipboard', None, _('Show _clipboard'), 'F4',
             _('Show the clipboard'), self._show_clipboard_cb, False),
            )

        # Create actions that reqiuire a project to be enabled
        bar_manager.add_actions(
            'ContextActions',
            # File menu
            ('Save', gtk.STOCK_SAVE, None, None,
             _('Save Project'), self._save_cb),
            ('SaveAs', gtk.STOCK_SAVE_AS, _('Save _As...'), '<shift><control>S',
             _('Save project with different name'), self._save_as_cb),
            ('Close', gtk.STOCK_CLOSE, None, None,
             _('Close Project'), self._close_cb),
            # Edit menu
            ('Undo', gtk.STOCK_UNDO, None, '<control>Z',
             _('Undo last action'), self._undo_cb),
            ('Redo', gtk.STOCK_REDO, None, '<shift><control>Z',
             _('Redo last action'), self._redo_cb)
            )

        bar_manager.add_actions(
            'AlwaysDisabled',
            # Edit menu
            ('Cut', gtk.STOCK_CUT, None, None,
             _('Cut'), None),
            ('Copy', gtk.STOCK_COPY, None, None,
             _('Copy'), None),
            ('Paste', gtk.STOCK_PASTE, None, None,
             _('Paste'), None),
            ('Delete', gtk.STOCK_DELETE, None, '<control>D',
             _('Delete'), None)
            )        
        
        bar_manager.build_interfaces()
        self._add_recent_items()
        application_window.add_accel_group(bar_manager.get_accel_group())
        main_vbox.pack_start(bar_manager.get_menubar(), False)
        main_vbox.pack_start(bar_manager.get_toolbar(), False)

        hbox = gtk.HBox(spacing=6)
        main_vbox.pack_start(hbox)
        
        palette.connect('toggled', self._palette_button_clicked)
        hbox.pack_start(palette, False, False)

        vpaned = gtk.VPaned()
        vpaned.set_position(150)
        hbox.pack_start(vpaned, True, True)

        notebook = gtk.Notebook()
        vpaned.add1(notebook)

        # Widget view
        widget_view = WidgetTreeView(self)
        self._add_view(widget_view)
        page_num = notebook.append_page(widget_view, gtk.Label(_('Widgets')))

        state = WidgetUIMState()
        self._uim_states[page_num] = state
        
        # Action view
        self.gactions_view = GActionsView(self)
        self._add_view(self.gactions_view)
        page_num = notebook.append_page(self.gactions_view, gtk.Label(_('Actions')))

        state = ActionUIMState(self.gactions_view)
        self._uim_states[page_num] = state

        # Sizegroup view
        self.sizegroup_view = SizeGroupView(self)
        self._add_view(self.sizegroup_view)
        page_num = notebook.append_page(self.sizegroup_view, gtk.Label(_('Size Groups')))

        state = SizeGroupUIMState(self.sizegroup_view)
        self._uim_states[page_num] = state

        # Add property editor
        self._editor = Editor(self)
        vpaned.add2(self._editor)

        notebook.connect('switch-page', self._switch_page_cb)

        # Statusbar
        statusbar = gtk.Statusbar()
        self._statusbar_menu_context_id = statusbar.get_context_id("menu")
        self._statusbar_actions_context_id = statusbar.get_context_id("actions")
        main_vbox.pack_end(statusbar, False)
        self._statusbar = statusbar
        
        # dnd doesn't seem to work with Konqueror, at least not when
        # gtk.DEST_DEFAULT_ALL or gtk.DEST_DEFAULT_MOTION is used. If
        # handling the drag-motion event it will work though, but it's
        # a bit tricky.
        application_window.drag_dest_set(gtk.DEST_DEFAULT_ALL,
                                         Application.targets,
                                         gtk.gdk.ACTION_COPY)
        
        application_window.connect('drag_data_received',
                                   self._dnd_data_received_cb)

        # Enable the current state
        self._active_uim_state = self._uim_states[0]
        self._active_uim_state.enable()
        
        return application_window

    def _add_recent_items(self):
        action_group = bar_manager.get_group('Normal')
        ui = ""
        for i, path in enumerate(config.recent_projects):
            if not os.path.exists(path):
                config.recent_projects.remove(path)
                continue
            
            basename = os.path.basename(path)
            if basename.endswith('.glade'):
                basename = basename[:-6]
                    
            ui += '<menuitem action="%s"/>' % basename
            label = '%d. %s' % (i+ 1, basename)
            action = gtk.Action(basename, label, '', '')
            action.connect('activate', self._open_project_cb, path)
            bar_manager.add_action('Normal', action)

        recent_template = '''<ui>
  <menubar name="MainMenu">
    <menu action="FileMenu">
      <placeholder name="RecentProjects">
      %s
      </placeholder>
    </menu>
  </menubar>
</ui>'''

        bar_manager.add_ui_from_string(recent_template % ui)

    def _add_view(self, view):
        self._project_views.insert(0, view)
        view.set_project(self._project)
        view.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)

    def _create_frame(self, label, widget):
        frame = gtk.Frame()
        label = gtk.Label('<span size="large" weight="bold">%s</span>' % label)
        label.set_use_markup(True)
        frame.set_label_widget(label)
        frame.set_shadow_type(gtk.SHADOW_NONE)
        frame.add(widget)
        return frame
    
    def _push_statusbar_hint(self, msg):
        self._statusbar.push(self._statusbar_menu_context_id, msg)

    def _pop_statusbar_hint(self):
        self._statusbar.pop(self._statusbar_menu_context_id)

    def _confirm_open_project(self, old_project, path):
        """Ask the user whether to reopen a project that has already
        been loaded (and thus reverting all changes) or just switch to
        the loaded version.
        """
        submsg1 = _('"%s" is already open.') % \
                  old_project.name
        submsg2 = _('Do you wish to revert your changes and re-open the '
                    'glade file? Your changes will be permanently lost '
                    'if you choose to revert.')
        msg = '<span weight="bold" size="larger">%s</span>\n\n%s\n' % \
              (submsg1, submsg2)
        dialog = gtk.MessageDialog(self.window, gtk.DIALOG_MODAL,
                                   gtk.MESSAGE_WARNING, gtk.BUTTONS_NONE,
                                   msg)
        dialog.set_title('')
        dialog.label.set_use_markup(True)
        dialog.add_buttons(gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                           gtk.STOCK_REVERT_TO_SAVED, gtk.RESPONSE_YES)
        dialog.set_default_response(gtk.RESPONSE_CANCEL)

        ret = dialog.run()
        if ret == gtk.RESPONSE_YES:
            # Close the old project
            old_project.selection_clear(False)
            
            for widget in old_project.widgets:
                widget.destroy()

            bar_manager.remove_ui(old_project.uim_id)
            self._projects.remove(old_project)

            # Open the project again
            project = Project.open(path, self)
            if project is not None:
                self._add_project(project)
        else:
            # No revert - just switch to the old project.
            self._set_project(old_project)
            
        dialog.destroy()

    def _add_project(self, project):
        # if the project was previously added, don't reload
        for prj in self._projects:
            if prj.path and prj.path == project.path:
                self._set_project(prj)
                return

        self._projects.insert(0, project)

        # add the project in the /Project menu
        project_action = gtk.Action(project.get_id(),
                                    project.name, project.name, '')
        
        project_action.connect('activate', self._set_project_cb, project)
        bar_manager.add_action('Project', project_action)

        project_ui = """
        <ui>
          <menubar name="MainMenu">
            <menu action="ProjectMenu">
              <menuitem action="%s"/>
            </menu>
          </menubar>
        </ui>
        """ % project.get_id()
        
        project.uim_id = bar_manager.add_ui_from_string(project_ui)

        # connect to the project signals so that the editor can be updated
        project.connect('selection_changed',
                         self._project_selection_changed_cb)
        project.connect('project_changed', self._project_changed_cb)
        project.undo_stack.connect('changed', self._refresh_command_stack_view_cb)

        self._set_project(project)
        config.add_recent_project(project.path)
        
    def _set_project(self, project):
        """
        Set the specified project as the current project.

        @param project: the project
        @type project: gazpacho.project.Project
        """
        if project is self._project:
            return

        if project and project not in self._projects:
            print ('Could not set project because it could not be found '
                   'in the list')
            return

        self._project = project

        for view in self._project_views:
            view.set_project(project)

        for state in self._uim_states.itervalues():
            state.set_project(project)

        self._refresh_title()
        self.refresh_command_stack_view()

        if project:
            # trigger the selection changed signal to update the editor
            self._project.selection_changed()

        self._update_palette_sensitivity()
        
    def _refresh_title(self):
        if self._project:
            title = '%s - Gazpacho' % self._project.name
            if self._project.changed:
                title = '*' + title
        else:
            title = 'Gazpacho'
        self.set_title(title)

    def _refresh_project_entry(self, project):
        """Update the project label in the project menu."""
        bar_manager.set_action_prop(project.get_id(), label=project.name)
        
    def _is_writable(self, filename):
        """
        Check if it is possible to write to the file. If the file
        doesn't exist it checks if it is possible to create it.
        """
        if os.access(filename, os.W_OK):
            return True

        path = os.path.dirname(filename)
        if not os.path.exists(filename) and os.access(path, os.W_OK):
            return True

        return False

    def _save(self, project, path):
        """ Internal save """
        if self._is_writable(path):
            project.save(path)
            self._refresh_project_entry(project)
            self._refresh_title()
            success = True
        else:
             submsg1 = _('Could not save project "%s"') % project.name
             submsg2 = _('Project "%s" could not be saved to %s. '
                         'Permission denied.') % \
                       (project.name, os.path.abspath(path))
             text = '<span weight="bold" size="larger">%s</span>\n\n%s\n' % \
                    (submsg1, submsg2)
             result = messagedialog(
                 gtk.MESSAGE_ERROR, text, parent=self.window,
                 buttons=gtk.BUTTONS_NONE,
                 additional_buttons=(gtk.STOCK_CANCEL,
                                     gtk.RESPONSE_CANCEL,
                                     gtk.STOCK_SAVE_AS,
                                     gtk.RESPONSE_YES))

             if result == gtk.RESPONSE_YES:
                 success = self._project_save_as(_('Save as'), project)
             else:
                 success = False

        return success
    
    def _project_save_as(self, title, project):
        filename = save(title, folder=config.lastdirectory)
        if not filename:
            return False

        config.set_lastdirectory(filename)
        
        return self._save(project, filename)

    def _confirm_close_project(self, project):
        """Show a dialog asking the user whether or not to save the
        project before closing them.

        Return False if the user has chosen not to close the project.
        """
        return self._confirm_close_projects([project])

    def _confirm_close_projects(self, projects):
        """Show a dialog listing the projects and ask whether the user
        wants to save them before closing them.

        Return False if the user has chosen not to close the projects.
        """
        if not projects:
            return True
        
        if len(projects) == 1:
            dialog = SingleCloseConfirmationDialog(projects[0], self.window)
        else:
            dialog = MultipleCloseConfirmationDialog(projects, self.window)

        ret = dialog.run()
        if ret == gtk.RESPONSE_YES:
            # Go through the chosen projects and save them. Ask for a
            # file name if necessary.
            close = True
            projects_to_save = dialog.get_projects()
            for project in projects_to_save:
                if project.path:
                    saved = self._save(project, project.path)
                else:
                    title = _('Save as')
                    saved = self._project_save_as(title, project)
                    # If the user has pressed cancel we abort everything.
                if not saved:
                    close = False
                    break
        elif ret == gtk.RESPONSE_NO:
            # The user has chosen not to save any projects.
            close = True
        else:
            # The user has cancel the close request.
            close = False

        dialog.destroy()
        return close


    def _palette_button_clicked(self, palette):
        adaptor = palette.current

        # adaptor may be None if the selector was pressed
        self._add_class = adaptor
        if adaptor and adaptor.is_toplevel():
            command_manager.create(adaptor, None, self._project)
            palette.unselect_widget()
            self._add_class = None

    def _project_selection_changed_cb(self, project):
        if self._project != project:
            self._set_project(project)
            return

        if self._editor:
            children = self._project.selection
            if len(children) == 1 and not isinstance(children[0], Placeholder):
                self._editor.display(Widget.from_widget(children[0]))
            else:
                self._editor.clear()
                        
    # debugging windows
    
    def _delete_event_for_debugging_window(self, window, event, action):
        # this will hide the window
        action.activate()
        # we don't want the window to be destroyed
        return True
    
    def _create_debugging_window(self, view, title, action):
        win = gtk.Window()
        win.set_title(title)
        win.set_transient_for(self.window)
        win.add(view)
        view.show_all()
        win.connect('delete-event',
                    self._delete_event_for_debugging_window, action)
        return win
    
    def _update_palette_sensitivity(self):
        """Update sensitivity of  UI elements (Palette) depending on
        the project list. If the list is empty we should unsensitive
        everything. Otherwise we should set sensitive to True.
        """
        if self._projects:
            palette.set_sensitive(True)
        else:
            palette.set_sensitive(False)


    # Public API

    def new_project(self):
        project = Project(True, self)
        self._add_project(project)
    
    def get_current_project(self):
        return self._project

    def get_projects(self):
        return self._projects

    def open_project(self, path):
        # Check if the project is loaded and ask the user what to do.
        for project in self._projects:
            if project.path and project.path == path:
                self._confirm_open_project(project, path)
                return project
        
        try:
            project = Project.open(path, self)
            if project is not None:
                self._add_project(project)
            return project
        except ParseError, e:
            submsg1 = _('The project could not be loaded')
            submsg2 = _('An error occurred while parsing the file "%s".') % \
                      os.path.abspath(path)
            msg = '<span weight="bold" size="larger">%s</span>\n\n%s\n' % \
                      (submsg1, submsg2)
            error(msg)
            self._update_palette_sensitivity()
        except IOError:
            if path in config.recent_projects:
                config.recent_projects.remove(path)
                #self._update_recent_items()
                
            submsg1 = _('The project could not be loaded')
            submsg2 = _('The file "%s" could not be opened') % \
                      os.path.abspath(path)
            msg = '<span weight="bold" size="larger">%s</span>\n\n%s\n' % \
                      (submsg1, submsg2)
            error(msg)
            self._update_palette_sensitivity()
            
    def close_current_project(self):
        """
        Close the current project. If there still are open projects
        after the current project has been closed we select one of
        them.
        """
        if not self._project:
            return
        
        self._project.selection_clear(False)
        self._project.selection_changed()
        
        for widget in self._project.widgets:
            widget.destroy()

        bar_manager.remove_ui(self._project.uim_id)
        self._projects.remove(self._project)

        next_project = None
        if self._projects:
            next_project = self._projects[0]

        self._set_project(next_project)

    def refresh_command_stack_view(self):
        # Update the command stack view
        if self._command_stack_window is not None:
            command_stack_view = self._command_stack_window.get_child()
            command_stack_view.update()
        
    def set_title(self, title):
        self.window.set_title(title)
        
    def show_message(self, msg, howlong=5000):
        self._statusbar.push(self._statusbar_menu_context_id, msg)
        def remove_msg():
            self._statusbar.pop(self._statusbar_menu_context_id)
            return False
        
        gobject.timeout_add(howlong, remove_msg)

    def show_all(self):
        self.window.show_all()
        
    def refresh_editor(self):
        self._editor.refresh()

    def create(self, type_name):
        adaptor = widget_registry.get_by_name(type_name)
        self._command_manager.create(adaptor, None, self._project)
        
    def run(self):
        """Display the window and run the application"""
        self.show_all()
        
        gtk.main()

    def validate_widget_names(self, project, show_error_dialogs=True):
        """Return True if all the widgets in project have valid names.

        A widget has a valid name if the name is not empty an unique.
        As soon as this function finds a widget with a non valid name it
        select it in the widget tree and returns False
        """

        widget_names = [widget.name for widget in project.widgets]
        for widget in project.widgets:
            # skip internal children (they have None as the name)
            if widget.name is None:
                continue
            
            if widget.name == '':
                if show_error_dialogs:
                    error(_("There is a widget with an empty name"))
                project.selection_set(widget, True)
                return False

            widget_names.remove(widget.name)

            if widget.name in widget_names:
                if show_error_dialogs:
                    msg = _('The name "%s" is used in more than one widget')
                    error(msg % widget.name)
                project.selection_set(widget, True)
                return False

        return True

    def get_current_context(self):
        """Return the context associated with the current project or None if
        there is not such a project.
        """
        if self._project is None:
            return None
        return self._project.context
    
    # Properties 

    def _get_window(self):
        return self._window
    window = property(_get_window)

    def get_add_class(self):
        return self._add_class
    add_class = property(get_add_class)
        
    # Callbacks

    def _delete_event_cb(self, window, event):
        self._quit_cb()

        # return TRUE to stop other handlers
        return True

    def _switch_page_cb(self, notebook, page, page_num):
        """
        Handler for the notebook's 'switch-page' signal. This will
        disable the current uim state, select a new and enable it
        instead.
        """
        # Disable old state
        self._active_uim_state.disable()

        # Select and enable new state
        state = self._uim_states[page_num]
        self._active_uim_state = state
        state.enable()

    def _refresh_command_stack_view_cb(self, undo_stack):
        self.refresh_command_stack_view()
    
    def _project_changed_cb(self, project):
        self._refresh_title()
        
    def _set_project_cb(self, action, project):
        self._set_project(project)

    def _open_project_cb(self, action, path):
        self.open_project(path)

    def _show_command_stack_cb(self, action):
        if self._command_stack_window is None:
            view = CommandStackView()
            self._add_view(view)
            
            title = _('Command Stack')
            action = bar_manager.get_action(
                '/MainMenu/DebugMenu/ShowCommandStack')
            self._command_stack_window = self._create_debugging_window(view,
                                                                       title,
                                                                       action)
            self._command_stack_window.show()
        else:
            if self._command_stack_window.get_property('visible'):
                self._command_stack_window.hide()
            else:
                self._command_stack_window.show_all()
                
    def _show_clipboard_cb(self, action):
        """Show/hide the clipboard window."""
        if self._clipboard_window is None:
            action = bar_manager.get_action(
                '/MainMenu/DebugMenu/ShowClipboard')
            self._clipboard_window = ClipboardWindow(self.window, clipboard)
            self._clipboard_window.connect(
                'delete-event', self._delete_event_for_debugging_window,
                action)
            self._clipboard_window.show_window()
        else:
            if self._clipboard_window.get_property('visible'):
                self._clipboard_window.hide_window()
            else:
                self._clipboard_window.show_window()

    def _dnd_data_received_cb(self, widget, context, x, y, data, info, time):
        """Callback that handles drag 'n' drop of glade files."""
        if info != Application.TARGET_TYPE_URI:
            return
        
        for uri in data.data.split('\r\n'):
            uri_parts = urlparse.urlparse(uri)
            if uri_parts[0] == 'file':
                path = urllib.url2pathname(uri_parts[2])
                self.open_project(path)

    # File action callbacks
    
    def _new_cb(self, action):
        self.new_project()

    def _open_cb(self, action):
        filename = open(parent=self.window, patterns=['*.glade'],
                        folder=config.lastdirectory)
        if filename:
            self.open_project(filename)
            config.set_lastdirectory(filename)
        
    def _save_cb(self, action):
        project = self._project

        # check that all the widgets have valid names
        if not self.validate_widget_names(project):
            return
                
        if project.path is not None:
            self._save(project, project.path)
            return

        # If instead we don't have a path yet, fire up a file chooser
        self._save_as_cb(None)

    def _save_as_cb(self, action):
        if action is None:
            # we were called from the _save_cb callback
            title = _('Save')
        else:
            title = _('Save as')
        self._project_save_as(title, self._project)

    def _close_cb(self, action):
        if not self._project:
            return

        if self._project.changed:
            close = self._confirm_close_project(self._project)
            if not close:
                return

        self.close_current_project()

    def _quit_cb(self, action=None):
        unsaved_projects = [p for p in self._projects if p.changed]
        close = self._confirm_close_projects(unsaved_projects)
        if not close:
            return

        config.save()
        gtk.main_quit()

    # Edit action callbacks
    
    def _undo_cb(self, action):
        command_manager.undo(self._project)
        self._editor.refresh()

    def _redo_cb(self, action):
        command_manager.redo(self._project)
        self._editor.refresh()
        
    def _show_structure_cb(self, action):
        self._show_structure = not self._show_structure
        for project in self._projects:
            for widget in project.widgets:
                if isinstance(widget, gtk.Window):
                    widget.queue_draw()
        
    # Debug action callbacks
    
    def _dump_data_cb(self, action):
        """This method is only useful for debugging.

        Any developer can print whatever he/she wants here
        the only rule is: clean everything before you commit it.

        This will be called upon CONTROL+M or by using the menu
        item Debug/Dump Data
        """
        project = self.get_current_project()
        print project.serialize()[:-1]

        def dump_widget(widget, lvl=0):
            print '%s %s (%s)' % (' ' * lvl,
                                  gobject.type_name(widget),
                                  widget.get_name())
            if isinstance(widget, gtk.Container):
                for child in widget.get_children():
                    dump_widget(child, lvl+1)

        for widget in project.widgets:
            if widget.get_parent():
                continue
            dump_widget(widget)
        
    def _reload_cb(self, action):
        gobject.idle_add(rebuild)

    # Help action callbacks
    
    def _about_cb(self, action):
        docs = environ.get_docs_dir()
        about = gtk.AboutDialog()
        about.set_name('Gazpacho')
        about.set_version(__version__)
        authorsfile = file(os.path.join(docs, 'AUTHORS'))
        authors = [a.strip() for a in authorsfile.readlines()]
        authors.append('') # separate authors from contributors
        contributorsfile = file(os.path.join(docs, 'CONTRIBUTORS'))
        authors.extend([c.strip() for c in contributorsfile.readlines()[:-2]])
        about.set_authors(authors)
        license = file(os.path.join(docs, 'COPYING')).read()
        about.set_license(license)
        about.set_website('http://gazpacho.sicem.biz')
        about.run()
        about.destroy()

    # Do not add anything here, add it above in the appropriate section
    
def register_stock_icons():
    
    # Register icons
    icon_factory = gtk.IconFactory()
    icon_set = gtk.IconSet()
    
    # "icon missing"
    icon_source = gtk.IconSource()
    icon_source.set_size_wildcarded(True)
    icon_source.set_filename(environ.find_pixmap('gazpacho-icon.png'))
    
    icon_set.add_source(icon_source)
   
    icon_factory.add(MISSING_ICON, icon_set)
    
    icon_factory.add_default()

if environ.epydoc:
    gazpacho = object()
else:
    register_stock_icons()
    gazpacho = Application()
