# Introduction

This document describes the `.build_config` files that are used by the
Chromium build system for Android-specific targets like APK, resources,
and more.

[TOC]

# I. Overview of .build_config files:

The Android build requires performing computations about dependencies in
various targets, which are not possible with the GN build language. To address
this, `.build_config` files are written during the build to store the needed
per-target information as JSON files.

They are always written to `$target_gen_dir/${target_name}.build_config`.

Many scripts under [`build/android/gyp/`](build/android_gyp/), which are used
during the build, can also accept parameter arguments using
`@FileArg references`, which look like:

    --some-param=@FileArg(<filename>:<key1>:<key2>:..<keyN>)

This placeholder will ensure that `<filename>` is read as a JSON file, then
return the value at `[key1][key2]...[keyN]` for the `--some-param` option.

Apart from that, the scripts do not need to know anything about the structure
of `.build_config` files (but the GN rules that invoke them do and select
which `@FileArg()` references to use).

For a concrete example, consider the following GN fragment:

```gn
# From //ui/android/BUILD.gn:
android_resources("ui_java_resources") {
  custom_package = "org.chromium.ui"
  resource_dirs = [ "java/res" ]
  deps = [
    ":ui_strings_grd",
  ]
}
```

This will end up generating the following JSON file under
`$CHROMIUM_OUTPUT_DIR/gen/ui/android/ui_java_resources.build_config`:

```json
{
  "deps_info": {
    "deps_configs": [
      "gen/ui/android/ui_strings_grd.build_config"
    ],
    "name": "ui_java_resources.build_config",
    "package_name": "org.chromium.ui",
    "path": "gen/ui/android/ui_java_resources.build_config",
    "r_text": "gen/ui/android/ui_java_resources_R.txt",
    "resources_dirs": [
      "../../ui/android/java/res"
    ],
    "resources_zip": "resource_zips/ui/android/ui_java_resources.resources.zip",
    "srcjar": "gen/ui/android/ui_java_resources.srcjar",
    "type": "android_resources"
  },
  "gradle": {},
  "resources": {
    "dependency_zips": [
      "resource_zips/ui/android/ui_strings_grd.resources.zip"
    ],
    "extra_package_names": [],
  }
}
```

NOTE: All path values in `.build_config` files are relative to your
`$CHROMIUM_OUTPUT_DIR`.

# II. Generation of .build_config files:

They are generated by the GN [`write_build_config()`](gn_write_build_config)
internal template, which ends up invoking
[`write_build_config.py`](write_build_config_py). For our example above, this
is with the following parameters:

```
python ../../build/android/gyp/write_build_config.py \
    --type=android_resources \
    --depfile gen/ui/android/ui_java_resources__build_config_crbug_908819.d \
    --deps-configs=\[\"gen/ui/android/ui_strings_grd.build_config\"\] \
    --build-config gen/ui/android/ui_java_resources.build_config \
    --resources-zip resource_zips/ui/android/ui_java_resources.resources.zip \
    --package-name org.chromium.ui \
    --r-text gen/ui/android/ui_java_resources_R.txt \
    --resource-dirs=\[\"../../ui/android/java/res\"\] \
    --srcjar gen/ui/android/ui_java_resources.srcjar
```

Note that *most* of the content of the JSON file comes from command-line
parameters, but not all of it.

In particular, the `resources['dependency_zips']` entry was computed by
inspecting the content of all dependencies (here, only
`ui_string_grd.build_config`), and collecting their
`deps_configs['resources_zip']` values.

Because a target's `.build_config` file will always be generated after
that of all of its dependencies,
[`write_build_config.py`](write_build_config_py) can traverse the
whole (transitive) set of direct *and* indirect dependencies for a given target
and extract useful information out of it.

This is the kind of processing that cannot be done at the GN language level,
and is very powerful for Android builds.


# III. Usage of .build_config files:

In addition to being parsed by `write_build_config.py`, when they are listed
in the `--deps-configs` of a given target, the `.build_config` files are used
by other scripts under [build/android/gyp/] to build stuff.

For example, the GN `android_resources` template uses it to invoke the
[`process_resources.py`] script with the following command, in order to
generate various related files (e.g. `ui_java_resources_R.txt`):

```sh
python ../../build/android/gyp/process_resources.py \
    --depfile gen/ui/android/ui_java_resources_1.d \
    --android-sdk-jar ../../third_party/android_sdk/public/platforms/android-29/android.jar \
    --aapt-path ../../third_party/android_sdk/public/build-tools/29.0.2/aapt \
    --dependencies-res-zips=@FileArg\(gen/ui/android/ui_java_resources.build_config:resources:dependency_zips\) \
    --extra-res-packages=@FileArg\(gen/ui/android/ui_java_resources.build_config:resources:extra_package_names\) \
    --resource-dirs=\[\"../../ui/android/java/res\"\] \
    --debuggable \
    --resource-zip-out resource_zips/ui/android/ui_java_resources.resources.zip \
    --r-text-out gen/ui/android/ui_java_resources_R.txt \
    --srcjar-out gen/ui/android/ui_java_resources.srcjar \
    --non-constant-id \
    --custom-package org.chromium.ui \
    --shared-resources
```

Note the use of `@FileArg()` references here, to tell the script where to find
the information it needs.


# IV. Format of .build_config files:

Thanks to `@FileArg()` references, Python build scripts under
[`build/android/gyp/`](build/android/gyp/)  do not need to know anything
about the internal format of `.build_config` files.

This format is decided between internal GN build rules and
[`write_build_config.py`][write_build_config_py]. Since these changes rather
often, the format documentation is kept inside the Python script itself, but
can be extracted as a Markdown file and visualized with the following commands:

```sh
# Extract .build_config format documentation
build/android/gyp/write_build_config.py \
  --generate-markdown-format-doc > /tmp/format.md

# Launch a browser to visualize the format documentation.
python tools/md_browser/md_browser.py -d /tmp /tmp/format.md
```

[build/android/gyp/]: https://chromium.googlesource.com/chromium/src/build/+/master/android/gyp/
[gn_write_build_config]: https://cs.chromium.org/chromium/src/build/config/android/internal_rules.gni?q=write_build_config&sq=package:chromium
[write_build_config_py]: https://chromium.googlesource.com/chromium/src/build/+/master/android/gyp/write_build_config.py
