// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/webui/side_panel/customize_chrome/customize_chrome_ui.h"

#include <string>
#include <utility>

#include "chrome/browser/cart/cart_handler.h"
#include "chrome/browser/new_tab_page/modules/new_tab_page_modules.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/search/background/ntp_custom_background_service_factory.h"
#include "chrome/browser/signin/identity_manager_factory.h"
#include "chrome/browser/ui/webui/new_tab_page/new_tab_page_ui.h"
#include "chrome/browser/ui/webui/sanitized_image_source.h"
#include "chrome/browser/ui/webui/side_panel/customize_chrome/customize_chrome_page_handler.h"
#include "chrome/browser/ui/webui/side_panel/customize_chrome/customize_chrome_section.h"
#include "chrome/browser/ui/webui/webui_util.h"
#include "chrome/common/webui_url_constants.h"
#include "chrome/grit/generated_resources.h"
#include "chrome/grit/side_panel_customize_chrome_resources.h"
#include "chrome/grit/side_panel_customize_chrome_resources_map.h"
#include "components/strings/grit/components_strings.h"
#include "content/public/browser/web_contents.h"
#include "content/public/browser/web_ui.h"
#include "content/public/browser/web_ui_data_source.h"
#include "third_party/abseil-cpp/absl/types/optional.h"
#include "ui/base/webui/web_ui_util.h"

DEFINE_CLASS_ELEMENT_IDENTIFIER_VALUE(CustomizeChromeUI,
                                      kChangeChromeThemeButtonElementId);
DEFINE_CLASS_ELEMENT_IDENTIFIER_VALUE(CustomizeChromeUI,
                                      kChangeChromeThemeClassicElementId);
DEFINE_CLASS_ELEMENT_IDENTIFIER_VALUE(CustomizeChromeUI,
                                      kChromeThemeBackElementId);
DEFINE_CLASS_ELEMENT_IDENTIFIER_VALUE(CustomizeChromeUI,
                                      kChromeThemeCollectionElementId);
DEFINE_CLASS_ELEMENT_IDENTIFIER_VALUE(CustomizeChromeUI, kChromeThemeElementId);

CustomizeChromeUI::CustomizeChromeUI(content::WebUI* web_ui)
    : ui::MojoBubbleWebUIController(web_ui),
      profile_(Profile::FromWebUI(web_ui)),
      web_contents_(web_ui->GetWebContents()),
      module_id_names_(ntp::MakeModuleIdNames(
          NewTabPageUI::IsDriveModuleEnabledForProfile(profile_))),
      page_factory_receiver_(this) {
  content::WebUIDataSource* source = content::WebUIDataSource::CreateAndAdd(
      profile_, chrome::kChromeUICustomizeChromeSidePanelHost);

  static constexpr webui::LocalizedString kLocalizedStrings[] = {
      // Side panel strings.
      {"backButton", IDS_ACCNAME_BACK},
      {"title", IDS_SIDE_PANEL_CUSTOMIZE_CHROME_TITLE},
      // Header strings.
      {"appearanceHeader", IDS_NTP_CUSTOMIZE_APPEARANCE_LABEL},
      {"cardsHeader", IDS_NTP_CUSTOMIZE_MENU_MODULES_LABEL},
      {"categoriesHeader", IDS_NTP_CUSTOMIZE_THEMES_HEADER},
      {"shortcutsHeader", IDS_NTP_CUSTOMIZE_MENU_SHORTCUTS_LABEL},
      // Appearance strings.
      {"changeTheme", IDS_NTP_CUSTOMIZE_CHROME_CHANGE_THEME_LABEL},
      {"chromeColors", IDS_NTP_CUSTOMIZE_CHROME_COLORS_LABEL},
      {"chromeWebStore", IDS_EXTENSION_WEB_STORE_TITLE},
      {"classicChrome", IDS_NTP_CUSTOMIZE_NO_BACKGROUND_LABEL},
      {"colorsContainerLabel", IDS_NTP_THEMES_CONTAINER_LABEL},
      {"colorPickerLabel", IDS_NTP_CUSTOMIZE_COLOR_PICKER_LABEL},
      {"currentTheme", IDS_NTP_CUSTOMIZE_CHROME_CURRENT_THEME_LABEL},
      {"defaultColorName", IDS_NTP_CUSTOMIZE_DEFAULT_LABEL},
      {"mainColorName", IDS_NTP_CUSTOMIZE_MAIN_COLOR_LABEL},
      {"managedColorsTitle", IDS_NTP_THEME_MANAGED_DIALOG_TITLE},
      {"managedColorsBody", IDS_NTP_THEME_MANAGED_DIALOG_BODY},
      {"uploadImage", IDS_NTP_CUSTOM_BG_UPLOAD_AN_IMAGE},
      {"uploadedImage", IDS_NTP_CUSTOMIZE_UPLOADED_IMAGE_LABEL},
      {"resetToClassicChrome",
       IDS_NTP_CUSTOMIZE_CHROME_RESET_TO_CLASSIC_CHROME_LABEL},
      {"refreshDaily", IDS_NTP_CUSTOM_BG_DAILY_REFRESH},
      // Shortcut strings.
      {"mostVisited", IDS_NTP_CUSTOMIZE_MOST_VISITED_LABEL},
      {"myShortcuts", IDS_NTP_CUSTOMIZE_MY_SHORTCUTS_LABEL},
      {"shortcutsCurated", IDS_NTP_CUSTOMIZE_MY_SHORTCUTS_DESC},
      {"shortcutsSuggested", IDS_NTP_CUSTOMIZE_MOST_VISITED_DESC},
      {"showShortcutsToggle", IDS_NTP_CUSTOMIZE_SHOW_SHORTCUTS_LABEL},
      // Card strings.
      {"showCardsToggleTitle", IDS_NTP_CUSTOMIZE_SHOW_CARDS_LABEL},
      {"modulesCartDiscountConsentAccept",
       IDS_NTP_MODULES_CART_DISCOUNT_CONSENT_ACCEPT},
      // Required by <managed-dialog>.
      {"controlledSettingPolicy", IDS_CONTROLLED_SETTING_POLICY},
      {"close", IDS_NEW_TAB_VOICE_CLOSE_TOOLTIP},
      {"ok", IDS_OK},
  };
  source->AddLocalizedStrings(kLocalizedStrings);

  source->AddBoolean(
      "modulesEnabled",
      ntp::HasModulesEnabled(module_id_names_,
                             IdentityManagerFactory::GetForProfile(profile_)));

  webui::SetupWebUIDataSource(
      source,
      base::make_span(kSidePanelCustomizeChromeResources,
                      kSidePanelCustomizeChromeResourcesSize),
      IDR_SIDE_PANEL_CUSTOMIZE_CHROME_CUSTOMIZE_CHROME_HTML);

  content::URLDataSource::Add(profile_,
                              std::make_unique<SanitizedImageSource>(profile_));
}

CustomizeChromeUI::~CustomizeChromeUI() = default;

void CustomizeChromeUI::ScrollToSection(CustomizeChromeSection section) {
  if (customize_chrome_page_handler_) {
    customize_chrome_page_handler_->ScrollToSection(section);
  } else {
    section_ = section;
  }
}

base::WeakPtr<CustomizeChromeUI> CustomizeChromeUI::GetWeakPtr() {
  return weak_ptr_factory_.GetWeakPtr();
}

WEB_UI_CONTROLLER_TYPE_IMPL(CustomizeChromeUI)

void CustomizeChromeUI::BindInterface(
    mojo::PendingReceiver<side_panel::mojom::CustomizeChromePageHandlerFactory>
        receiver) {
  if (page_factory_receiver_.is_bound()) {
    page_factory_receiver_.reset();
  }
  page_factory_receiver_.Bind(std::move(receiver));
}

void CustomizeChromeUI::BindInterface(
    mojo::PendingReceiver<chrome_cart::mojom::CartHandler>
        pending_page_handler) {
  cart_handler_ = std::make_unique<CartHandler>(std::move(pending_page_handler),
                                                profile_, web_contents_);
}

void CustomizeChromeUI::BindInterface(
    mojo::PendingReceiver<help_bubble::mojom::HelpBubbleHandlerFactory>
        pending_receiver) {
  if (help_bubble_handler_factory_receiver_.is_bound()) {
    help_bubble_handler_factory_receiver_.reset();
  }
  help_bubble_handler_factory_receiver_.Bind(std::move(pending_receiver));
}

void CustomizeChromeUI::CreatePageHandler(
    mojo::PendingRemote<side_panel::mojom::CustomizeChromePage> pending_page,
    mojo::PendingReceiver<side_panel::mojom::CustomizeChromePageHandler>
        pending_page_handler) {
  DCHECK(pending_page.is_valid());
  customize_chrome_page_handler_ = std::make_unique<CustomizeChromePageHandler>(
      std::move(pending_page_handler), std::move(pending_page),
      NtpCustomBackgroundServiceFactory::GetForProfile(profile_), web_contents_,
      module_id_names_);
  if (section_.has_value()) {
    customize_chrome_page_handler_->ScrollToSection(*section_);
    section_.reset();
  }
}

void CustomizeChromeUI::CreateHelpBubbleHandler(
    mojo::PendingRemote<help_bubble::mojom::HelpBubbleClient> client,
    mojo::PendingReceiver<help_bubble::mojom::HelpBubbleHandler> handler) {
  help_bubble_handler_ = std::make_unique<user_education::HelpBubbleHandler>(
      std::move(handler), std::move(client), this,
      std::vector<ui::ElementIdentifier>{
          CustomizeChromeUI::kChangeChromeThemeButtonElementId,
          CustomizeChromeUI::kChangeChromeThemeClassicElementId,
          CustomizeChromeUI::kChromeThemeCollectionElementId,
          CustomizeChromeUI::kChromeThemeElementId,
          CustomizeChromeUI::kChromeThemeBackElementId});
}
