// Copyright (C) 2022 The Qt Company Ltd.
// Copyright (C) 2019 Alexey Edelev <semlanik@gmail.com>, Tatyana Borisova <tanusshhka@mail.ru>
// SPDX-License-Identifier: LicenseRef-Qt-Commercial OR GPL-3.0-only WITH Qt-GPL-exception-1.0

#include "commontemplates.h"

// CommonTemplates navigation map:
// All template functions are named as following(with some exceptions):
//     <CommonTemplateType>[TemplateSubject][Declaration|Definition]Template
//
// CommonTemplateType usually describes what template implements e.g. Constructor, Member,
// Property etc.
//
// TemplateSubject is the type that the template is applicable for. The following naming is used for
// the common types:
// ''(empty) - for integral types like int32, fixed32, bool, etc.
// Map       - any map types.
// Repeated  - any repeated types.
// Complex   - means that template is applicable for any non-trivially desctructable types, except
//             messages.
// Message   - for any general protobuf message that is represented as the pointer with lazy
//             initialization
// Enum      - any enum types.
// Optional  - any field that behaves like optional field.
//
// Map and Repeated can be combined with one of Complex, Message or Enum keywords, that indicates
// the specialization of the template.

using namespace ::qtprotoccommon;

const char *CommonTemplates::ProtoSuffix()
{
    return "_proto";
}

const std::vector<std::string> &CommonTemplates::ListOfQmlExceptions()
{
    static std::vector<std::string> vec{ { "id", "property", "import" } };
    return vec;
}

const char *CommonTemplates::DefaultProtobufIncludesTemplate()
{
    return "#include <QtCore/QMetaType>\n"
           "#include <QtCore/QList>\n"
           "\n"
           "#include <QtProtobuf/qprotobufobject.h>\n"
           "#include <QtProtobuf/qprotobuflazymessagepointer.h>\n"
           "\n"
           "#include <memory>\n"
           "\n";
}

const char *CommonTemplates::QmlProtobufIncludesTemplate()
{
    return "#include <QtQml/QQmlListProperty>\n"
           "#include <QtProtobufQuick/qtprotobufqmllistpropertyconstructor.h>\n\n";
}

const char *CommonTemplates::DisclaimerTemplate()
{
    return "/* This file is autogenerated. DO NOT CHANGE. All changes will be lost */\n\n";
}

const char *CommonTemplates::PreambleTemplate()
{
    return "#ifndef Q_PROTOBUF_$filename$_H\n"
           "#define Q_PROTOBUF_$filename$_H\n\n"
           "#include <QtProtobuf/QProtobufMessage>\n";
}

const char *CommonTemplates::FooterTemplate()
{
    return "#endif // Q_PROTOBUF_$filename$_H\n";
}

const char *CommonTemplates::InternalIncludeTemplate()
{
    return "#include \"$include$.h\"\n";
}
const char *CommonTemplates::ExternalIncludeTemplate()
{
    return "#include <$include$>\n";
}
const char *CommonTemplates::GlobalEnumIncludeTemplate()
{
    return "#include <globalenums.h>\n";
}

const char *CommonTemplates::EnumRegistrationDeclaration()
{
    return "$export_macro$ void registerTypes();\n";
}

const char *CommonTemplates::MetaTypeRegistrationDeclaration()
{
    return "static void registerTypes();\n";
}
const char *CommonTemplates::MetaTypeRegistrationMessageDefinition()
{
    return "void $type$::registerTypes()\n{\n"
           "    qRegisterMetaType<$type$>();\n"
           "    qRegisterMetaType<$list_type$>();\n";
}
const char *CommonTemplates::MetaTypeRegistrationGlobalEnumDefinition()
{
    return "void $enum_gadget$::registerTypes()\n{\n";
}
const char *CommonTemplates::MetaTypeRegistrationGlobalEnumTemplate()
{
    return "qRegisterMetaType<$type$>();\n"
           "qRegisterMetaType<$full_type$>();\n";
}
const char *CommonTemplates::UsingMessageTemplate()
{
    return "using $classname$Repeated = QList<std::shared_ptr<$classname$>>;\n";
}
const char *CommonTemplates::UsingMapTemplate()
{
    return "using $type$ = QHash<$key_type$, $value_type$>;\n";
}
const char *CommonTemplates::UsingMapMessageTemplate()
{
    return "using $type$ = QHash<$key_type$, std::shared_ptr<$value_type$>>;\n";
}
const char *CommonTemplates::UsingNestedMessageTemplate()
{
    return "using $type$ = $scope_namespaces$::$type$;\n"
           "using $list_type$ = $scope_namespaces$::$list_type$;\n";
}

const char *CommonTemplates::UsingRepeatedEnumTemplate()
{
    return "using $list_type$ = QList<$type$>;\n";
}

const char *CommonTemplates::NamespaceTemplate()
{
    return "namespace $scope_namespaces$ {\n";
}
const char *CommonTemplates::NamespaceClosingTemplate()
{
    return "} // namespace $scope_namespaces$\n";
}
const char *CommonTemplates::EnumDeclarationTemplate()
{
    return "\nnamespace $classname$ {\n"
           "Q_NAMESPACE_EXPORT($export_macro$)\n";
}
const char *CommonTemplates::ClassMessageForwardDeclarationTemplate()
{
    return "class $classname$;\n";
}
const char *CommonTemplates::ClassMessageBeginDeclarationTemplate()
{
    return "\nclass $export_macro$ $classname$ : public QProtobufMessage\n"
           "{\n"
           "    Q_GADGET\n"
           "    Q_PROTOBUF_OBJECT\n"
           "    Q_DECLARE_PROTOBUF_SERIALIZERS($classname$)\n";
}

const char *CommonTemplates::PropertyTemplate()
{
    return "Q_PROPERTY($property_type$ $property_name$ READ $property_name$ WRITE "
           "set$property_name_cap$ SCRIPTABLE $scriptable$)\n";
}
const char *CommonTemplates::PropertyRepeatedTemplate()
{
    return "Q_PROPERTY($property_list_type$ $property_name$ READ $property_name$ WRITE "
           "set$property_name_cap$ SCRIPTABLE $scriptable$)\n";
}
const char *CommonTemplates::PropertyRepeatedMessageTemplate()
{
    return "Q_PROPERTY($property_list_type$ $property_name$Data READ $property_name$ WRITE "
           "set$property_name_cap$ SCRIPTABLE $scriptable$)\n";
}
const char *CommonTemplates::PropertyNonScriptableTemplate()
{
    return "Q_PROPERTY($property_type$ $property_name$_p READ $property_name$ WRITE "
           "set$property_name_cap$ SCRIPTABLE false)\n";
}
const char *CommonTemplates::PropertyNonScriptableAliasTemplate()
{
    return "Q_PROPERTY($qml_alias_type$ $property_name$ READ $property_name$_p WRITE "
           "set$property_name_cap$_p SCRIPTABLE true)\n";
}
const char *CommonTemplates::PropertyMessageTemplate()
{
    return "Q_PROPERTY($property_type$ *$property_name$ READ $property_name$_p WRITE "
           "set$property_name_cap$_p)\n";
}
const char *CommonTemplates::PropertyQmlListTemplate()
{
    return "Q_PROPERTY(QQmlListProperty<$property_type$> $property_name$ READ $property_name$_l)\n";
}

const char *CommonTemplates::ConstructorMessageDeclarationTemplate()
{
    return "$classname$();\n";
}

const char *CommonTemplates::DestructorMessageDeclarationTemplate()
{
    return "~$classname$();\n";
}

const char *CommonTemplates::MemberTemplate()
{
    return "$scope_type$ m_$property_name$;\n";
}
const char *CommonTemplates::MemberRepeatedTemplate()
{
    return "$scope_list_type$ m_$property_name$;\n";
}
const char *CommonTemplates::MemberMessageTemplate()
{
    return "QtProtobufPrivate::QProtobufLazyMessagePointer<$scope_type$> m_$property_name$;\n";
}
const char *CommonTemplates::PublicBlockTemplate()
{
    return "\npublic:\n";
}
const char *CommonTemplates::PrivateBlockTemplate()
{
    return "\nprivate:\n";
}
const char *CommonTemplates::EnumDefinitionTemplate()
{
    return "enum $type$ {\n";
}
const char *CommonTemplates::EnumFieldTemplate()
{
    return "$enumvalue$ = $value$,\n";
}

const char *CommonTemplates::ConstructorMessageDefinitionTemplate()
{
    return "$type$::$type$() : QProtobufMessage(&$type$::staticMetaObject)";
}

const char *CommonTemplates::EmptyConstructorTemplate()
{
    return "$classname$() {}\n";
}

const char *CommonTemplates::UseNamespace()
{
    return "using namespace $namespace$;\n";
}

const char *CommonTemplates::QObjectConstructorMessageDeclarationTemplate()
{
    return "explicit $classname$(QObject *parent = nullptr);\n";
}

const char *CommonTemplates::CopyConstructorDeclarationTemplate()
{
    return "$classname$(const $classname$ &other);\n";
}
const char *CommonTemplates::MoveConstructorDeclarationTemplate()
{
    return "$classname$($classname$ &&other) noexcept;\n";
}
const char *CommonTemplates::CopyConstructorDefinitionTemplate()
{
    return "$classname$::$classname$(const $classname$ &other) : "
           "QProtobufMessage(other)";
}
const char *CommonTemplates::MoveConstructorDefinitionTemplate()
{
    return "$classname$::$classname$($classname$ &&other) noexcept : "
           "QProtobufMessage(std::move(other))";
}
const char *CommonTemplates::DeletedCopyConstructorTemplate()
{
    return "$classname$(const $classname$ &) = delete;\n";
}
const char *CommonTemplates::DeletedMoveConstructorTemplate()
{
    return "$classname$($classname$ &&) = delete;\n";
}
const char *CommonTemplates::CopyMemberTemplate()
{
    return "set$property_name_cap$(other.m_$property_name$);\n";
}
const char *CommonTemplates::CopyMemberMessageTemplate()
{
    return "if (m_$property_name$ != other.m_$property_name$) {\n"
           "    *m_$property_name$ = *other.m_$property_name$;\n"
           "}\n";
}
const char *CommonTemplates::AssignMemberMessageTemplate()
{
    return "if (m_$property_name$ != other.m_$property_name$)\n"
           "    *m_$property_name$ = *other.m_$property_name$;\n";
}
const char *CommonTemplates::MoveMemberMessageTemplate()
{
    return "if (m_$property_name$ != other.m_$property_name$) {\n"
           "    *m_$property_name$ = std::move(*other.m_$property_name$);\n"
           "}\n";
}
const char *CommonTemplates::MoveAssignMemberMessageTemplate()
{
    return "if (m_$property_name$ != other.m_$property_name$)\n"
           "    *m_$property_name$ = std::move(*other.m_$property_name$);\n";
}
const char *CommonTemplates::MoveAssignMemberComplexTemplate()
{
    return "if (m_$property_name$ != other.m_$property_name$)\n"
           "    m_$property_name$ = std::move(other.m_$property_name$);\n";
}

const char *CommonTemplates::MoveConstructorMemberComplexTemplate()
{
    return "m_$property_name$ = std::move(other.m_$property_name$);\n";
}

const char *CommonTemplates::MoveMemberTemplate()
{
    return "set$property_name_cap$(std::exchange(other.m_$property_name$, 0));\n";
}
const char *CommonTemplates::MoveMemberEnumTemplate()
{
    return "m_$property_name$ = other.m_$property_name$;\n";
}

const char *CommonTemplates::AssignmentOperatorDeclarationTemplate()
{
    return "$classname$ &operator =(const $classname$ &other);\n";
}
const char *CommonTemplates::AssignmentOperatorDefinitionTemplate()
{
    return "$classname$ &$classname$::operator =(const $classname$ &other)\n"
           "{\n"
           "    QProtobufMessage::operator=(other);\n";
}
const char *CommonTemplates::AssignmentOperatorReturnTemplate()
{
    return "return *this;\n";
}

const char *CommonTemplates::MoveAssignmentOperatorDeclarationTemplate()
{
    return "$classname$ &operator =($classname$ &&other) noexcept;\n";
}
const char *CommonTemplates::MoveAssignmentOperatorDefinitionTemplate()
{
    return "$classname$ &$classname$::operator =($classname$ &&other) noexcept\n"
           "{\n"
           "    QProtobufMessage::operator=(std::move(other));\n";
}

const char *CommonTemplates::EqualOperatorDeclarationTemplate()
{
    return "bool operator ==(const $classname$ &other) const;\n";
}
const char *CommonTemplates::EqualOperatorDefinitionTemplate()
{
    return "bool $classname$::operator ==(const $classname$ &other) const\n{\n"
           "    return QProtobufMessage::isEqual(*this, other)";
}
const char *CommonTemplates::EqualOperatorMemberTemplate()
{
    return "m_$property_name$ == other.m_$property_name$";
}
const char *CommonTemplates::EqualOperatorMemberMessageTemplate()
{
    return "(m_$property_name$ == other.m_$property_name$\n"
           "    || *m_$property_name$ == *other.m_$property_name$)\n";
}
const char *CommonTemplates::EqualOperatorMemberRepeatedTemplate()
{
    return "QtProtobuf::repeatedValueCompare(m_$property_name$, other.m_$property_name$)";
}

const char *CommonTemplates::NotEqualOperatorDeclarationTemplate()
{
    return "bool operator !=(const $classname$ &other) const;\n";
}
const char *CommonTemplates::NotEqualOperatorDefinitionTemplate()
{
    return "bool $classname$::operator !=(const $classname$ &other) const\n{\n"
           "    return !this->operator ==(other);\n"
           "}\n\n";
}

const char *CommonTemplates::PrivateGetterMessageDeclarationTemplate()
{
    return "$getter_type$ *$property_name$_p() const;\n";
}
const char *CommonTemplates::PrivateGetterMessageDefinitionTemplate()
{
    return "$getter_type$ *$classname$::$property_name$_p() const\n{\n"
           "    return m_$property_name$ ? m_$property_name$.get() : nullptr;\n"
           "}\n\n";
}

const char *CommonTemplates::GetterMessageDeclarationTemplate()
{
    return "$getter_type$ &$property_name$() const;\n";
}
const char *CommonTemplates::GetterMessageDefinitionTemplate()
{
    return "$getter_type$ &$classname$::$property_name$() const\n{\n"
           "    return *m_$property_name$;\n"
           "}\n\n";
}

const char *CommonTemplates::GetterTemplate()
{
    return "$getter_type$ $property_name$() const {\n"
           "    return m_$property_name$;\n"
           "}\n\n";
}

const char *CommonTemplates::GetterNonScriptableTemplate()
{
    return "$qml_alias_type$ $property_name$_p() const {\n"
           "    return m_$property_name$;\n"
           "}\n\n";
}

const char *CommonTemplates::GetterComplexTemplate()
{
    return "$getter_type$ &$property_name$() {\n"
           "    return m_$property_name$;\n"
           "}\n\n";
}

const char *CommonTemplates::GetterQmlListDeclarationTemplate()
{
    return "QQmlListProperty<$scope_type$> $property_name$_l();\n";
}
const char *CommonTemplates::GetterQmlListDefinitionTemplate()
{
    return "QQmlListProperty<$full_type$> $classname$::$property_name$_l()\n{\n"
           "    return qProtobufConstructQmlListProperty<$scope_type$>(this, &m_$property_name$);\n"
           "}\n\n";
}

const char *CommonTemplates::PrivateSetterMessageDeclarationTemplate()
{
    return "void set$property_name_cap$_p($setter_type$ *$property_name$);\n";
}
const char *CommonTemplates::PrivateSetterMessageDefinitionTemplate()
{
    return "void $classname$::set$property_name_cap$_p($setter_type$ *$property_name$)\n{\n"
           "    if (m_$property_name$.get() != $property_name$)\n"
           "        m_$property_name$.reset($property_name$);\n"
           "}\n\n";
}

const char *CommonTemplates::SetterMessageDeclarationTemplate()
{
    return "void set$property_name_cap$(const $setter_type$ &$property_name$);\n";
}
const char *CommonTemplates::SetterMessageDefinitionTemplate()
{
    return "void $classname$::set$property_name_cap$(const $setter_type$ &$property_name$)\n{\n"
           "    if (*m_$property_name$ != $property_name$)\n"
           "        *m_$property_name$ = $property_name$;\n"
           "}\n\n";
}

const char *CommonTemplates::SetterComplexDeclarationTemplate()
{
    return "void set$property_name_cap$(const $setter_type$ &$property_name$);\n";
}
const char *CommonTemplates::SetterComplexDefinitionTemplate()
{
    return "void $classname$::set$property_name_cap$(const $setter_type$ &$property_name$)\n{\n"
           "    if (m_$property_name$ != $property_name$)\n"
           "        m_$property_name$ = $property_name$;\n"
           "}\n\n";
}

const char *CommonTemplates::SetterTemplate()
{
    return "void set$property_name_cap$(const $setter_type$ &$property_name$) {\n"
           "    if (m_$property_name$ != $property_name$)\n"
           "        m_$property_name$ = $property_name$;\n"
           "}\n\n";
}
const char *CommonTemplates::SetterNonScriptableTemplate()
{
    return "void set$property_name_cap$_p(const $qml_alias_type$ &$property_name$) {\n"
           "    if (m_$property_name$ != $property_name$)\n"
           "        m_$property_name$ = $property_name$;\n"
           "}\n\n";
}

const char *CommonTemplates::SignalsBlockTemplate()
{
    return "\nQ_SIGNALS:\n";
}
const char *CommonTemplates::SignalTemplate()
{
    return "void $property_name$Changed();\n";
}

const char *CommonTemplates::MetaDataUintDataEntryTemplate()
{
    return "const std::array<uint, $size$> qt_protobuf_$classname$_uint_data =";
}

const char *CommonTemplates::MetaDataCharDataEntryTemplate()
{
    return "const char qt_protobuf_$classname$_char_data[$size$] =";
}

const char *CommonTemplates::JsonNameOffsetsUintDataTemplate()
{
    return "$json_name_offset$, /* = $json_name$ */\n";
}

const char *CommonTemplates::FieldNumbersUintDataTemplate()
{
    return "$field_number$, /* = $json_name$ */\n";
}

const char *CommonTemplates::QtPropertyIndicesUintDataTemplate()
{
    return "$property_index$, /* = $json_name$ */\n";
}

const char *CommonTemplates::FieldFlagsUintDataTemplate()
{
    return "$field_flags$, /* = $json_name$ */\n";
}

const char *CommonTemplates::PropertyOrderingDataOpeningTemplate()
{
    return "static constexpr struct {\n"
           "    QtProtobufPrivate::QProtobufPropertyOrdering::Data data;\n"
           "    const std::array<uint, $uint_size$> qt_protobuf_$classname$_uint_data;\n"
           "    const char qt_protobuf_$classname$_char_data[$char_size$];\n"
           "} qt_protobuf_$classname$_metadata {\n"
           "    // data\n"
           "    {\n"
           "        $version_number$, /* = version */\n"
           "        $num_fields$, /* = num fields */\n"
           "        $field_number_offset$, /* = field number offset */\n"
           "        $property_index_offset$, /* = property index offset */\n"
           "        $field_flags_offset$, /* = field flags offset */\n"
           "        $message_full_name_size$, /* = message full name length */\n"
           "    },\n";
}

const char *CommonTemplates::PropertyOrderingDataClosingTemplate()
{
    return "};\n\n";
}
const char *CommonTemplates::PropertyOrderingDefinitionTemplate()
{
    return "const QtProtobufPrivate::QProtobufPropertyOrdering $type$::propertyOrdering = {\n"
           "    &qt_protobuf_$classname$_metadata.data\n"
           "};\n\n";
}

const char *CommonTemplates::SimpleBlockEnclosureTemplate()
{
    return "}\n";
}
const char *CommonTemplates::SemicolonBlockEnclosureTemplate()
{
    return "};\n";
}
const char *CommonTemplates::InitializerMemberTemplate()
{
    return "m_$property_name$($initializer$)";
}
const char *CommonTemplates::InitializerMemberMessageTemplate()
{
    return "m_$property_name$(nullptr)";
}
const char *CommonTemplates::CopyInitializerMemberTemplate()
{
    return "m_$property_name$(other.m_$property_name$)";
}
const char *CommonTemplates::EmptyBracesTemplate()
{
    return "\n{\n}\n";
}

const char *CommonTemplates::DeclareMetaTypeTemplate()
{
    return "Q_DECLARE_METATYPE($full_type$)\n";
}

const char *CommonTemplates::DeclareMetaTypeQmlListTemplate()
{
    return "Q_DECLARE_METATYPE(QQmlListProperty<$full_type$>)\n";
}

const char *CommonTemplates::MetaTypeRegistrationLocalEnumTemplate()
{
    return "qRegisterProtobufEnumType<$scope_type$>();\n"
           "qRegisterMetaType<$scope_type$>();\n"
           "qRegisterMetaType<$scope_type$>();\n"
           "qRegisterMetaType<$scope_list_type$>();\n";
}
const char *CommonTemplates::MetaTypeRegistrationMapTemplate()
{
    return "qRegisterMetaType<$scope_type$>();\n"
           "qRegisterProtobufMapType<$key_type$, $value_type$>();\n";
}

const char *CommonTemplates::MetaTypeRegistrationQmlListTemplate()
{
    return "qRegisterMetaType<QQmlListProperty<$full_type$>>();\n";
}

const char *CommonTemplates::QEnumTemplate()
{
    return "Q_ENUM($type$)\n";
}

const char *CommonTemplates::QEnumNSTemplate()
{
    return "Q_ENUM_NS($type$)\n";
}

const char *CommonTemplates::RegisterEnumSerializersTemplate()
{
    return "qRegisterProtobufEnumType<$full_type$>();\n";
}
const char *CommonTemplates::RegistrarTemplate()
{
    return "static QtProtobuf::ProtoTypeRegistrar "
           "ProtoTypeRegistrar$classname$(qRegisterProtobufType<$classname$>);\n";
}
const char *CommonTemplates::RegistrarEnumTemplate()
{
    return "static QtProtobuf::ProtoTypeRegistrar "
           "ProtoTypeRegistrar$enum_gadget$($enum_gadget$::registerTypes);\n";
}
const char *CommonTemplates::QmlRegisterTypeTemplate()
{
    return "qmlRegisterType<$scope_type$>(\"$qml_package$\", 1, 0, \"$type$\");\n";
}
const char *CommonTemplates::QmlRegisterTypeEnumTemplate()
{
    return "qmlRegisterUncreatableType<$enum_gadget$>(\"$qml_package$\", 1, 0, \"$type$\", "
           "QStringLiteral(\"$full_type$ Could not be created from qml context\"));\n";
}

const char *CommonTemplates::RepeatedSuffix()
{
    return "Repeated";
}

// Those marked "Limited" have limited usage in QML, since QML only supports signed integers.
// See https://doc.qt.io/qt-6/qtqml-typesystem-valuetypes.html for types that are supported by the
// QML JS engine.
const std::unordered_map<::google::protobuf::FieldDescriptor::Type, std::string>
        &CommonTemplates::TypeReflection()
{
    static std::unordered_map<::google::protobuf::FieldDescriptor::Type, std::string> map{
        { ::google::protobuf::FieldDescriptor::TYPE_DOUBLE, "double" },
        { ::google::protobuf::FieldDescriptor::TYPE_FLOAT, "float" },
        { ::google::protobuf::FieldDescriptor::TYPE_INT64, "int64" }, // Limited
        { ::google::protobuf::FieldDescriptor::TYPE_UINT64, "uint64" }, // Limited
        { ::google::protobuf::FieldDescriptor::TYPE_INT32, "int32" },
        { ::google::protobuf::FieldDescriptor::TYPE_FIXED64, "fixed64" }, // Limited
        { ::google::protobuf::FieldDescriptor::TYPE_FIXED32, "fixed32" },
        { ::google::protobuf::FieldDescriptor::TYPE_BOOL, "bool" },
        { ::google::protobuf::FieldDescriptor::TYPE_STRING, "QString" },
        { ::google::protobuf::FieldDescriptor::TYPE_BYTES, "QByteArray" },
        { ::google::protobuf::FieldDescriptor::TYPE_UINT32, "uint32" }, // Limited
        { ::google::protobuf::FieldDescriptor::TYPE_SFIXED32, "sfixed32" },
        { ::google::protobuf::FieldDescriptor::TYPE_SFIXED64, "sfixed64" }, // Limited
        { ::google::protobuf::FieldDescriptor::TYPE_SINT32, "sint32" },
        { ::google::protobuf::FieldDescriptor::TYPE_SINT64, "sint64" } // Limited
    };
    return map;
}

const char *CommonTemplates::ProtoFileSuffix()
{
    return ".qpb";
}

const char *CommonTemplates::EnumClassSuffix()
{
    return "Gadget";
}

const char *CommonTemplates::QtProtobufNamespace()
{
    return "QtProtobuf";
}
const char *CommonTemplates::QtProtobufNestedNamespace()
{
    return "_QtProtobufNested";
}

const char *CommonTemplates::QtProtobufFieldEnum()
{
    return "QtProtobufFieldEnum";
}
const char *CommonTemplates::FieldEnumTemplate()
{
    return "enum QtProtobufFieldEnum {\n";
}
const char *CommonTemplates::FieldNumberTemplate()
{
    return "$property_name_cap$ProtoFieldNumber = $number$,\n";
}

const char *CommonTemplates::ExportMacroTemplate()
{
    return "#if defined(QT_SHARED) || !defined(QT_STATIC)\n"
           "#  if defined(QT_BUILD_$export_macro$_LIB)\n"
           "#    define QPB_$export_macro$_EXPORT Q_DECL_EXPORT\n"
           "#  else\n"
           "#    define QPB_$export_macro$_EXPORT Q_DECL_IMPORT\n"
           "#  endif\n"
           "#else\n"
           "#  define QPB_$export_macro$_EXPORT\n"
           "#endif\n";
}

