/*
   Subdir.c - sub directory manipulation code.
   Copyright (C) 2000 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at:  imre.leber@worldonline.be
*/


#include <string.h>
#include <stdlib.h>

#include "..\header\rdwrsect.h"
#include "..\header\direct.h"
#include "..\header\bool.h"
#include "..\header\boot.h"
#include "..\header\fat.h"
#include "..\header\subdir.h"
#include "..\header\fatconst.h"

struct PipeStruct
{
       int    (*func) (RDWRHandle handle, struct DirectoryPosition* direct,
		       void** buffer);
       void** buffer;
       int    exact;
       int    error;
};

static int SubdirTraverser(RDWRHandle handle,
			   CLUSTER label,
			   SECTOR datasector,
			   void** structure)
{
    int i;
    struct DirectoryPosition dirpos;
    struct DirectoryEntry   direct;

    struct PipeStruct** pipe = (struct PipeStruct**) structure;

    dirpos.sector = (SECTOR) datasector;

    for (i = 0; i < 16; i++)
    {
	dirpos.offset = i;

	if ((*pipe)->exact)
	{
	   if (!GetDirectory(handle, &dirpos, &direct))
	   {
	      (*pipe)->error = 1;  
	      return FALSE;
	   }

	   if (direct.filename[0] == LASTLABEL)
	      return FALSE;
	}

	if (!(*pipe)->func(handle, &dirpos, (*pipe)->buffer))
	   return FALSE;
    }
    return TRUE;
}

int TraverseRootDir(RDWRHandle handle,
		    int (*func) (RDWRHandle handle,
				 struct DirectoryPosition* pos,
				 void** buffer),
		    void** buffer, int exact)
{
     int            i, j=0;
     unsigned short entries;

     struct DirectoryPosition dirpos;
     struct DirectoryEntry   direct;
	       
     entries = GetNumberOfRootEntries(handle);
     if (entries == 0) return FALSE;

     if (GetRootDirPosition(handle, 0, &dirpos) == FALSE) return FALSE;

     for (i = 0; i < entries; i++, j++)
     {
	 if (j == 16)
	 {
	    dirpos.sector++;
	    j = 0;
	 }
	 dirpos.offset = j;

	 if (exact)
	 {
	    if (!GetDirectory(handle, &dirpos, &direct)) return FALSE;
	    if (direct.filename[0] == LASTLABEL) return TRUE;
	 }

	 if (!func(handle, &dirpos, buffer)) return TRUE;
     }

     return TRUE;
}

int TraverseSubdir(RDWRHandle handle, CLUSTER fatcluster,
		   int (*func) (RDWRHandle handle,
				struct DirectoryPosition* pos,
				void** buffer),
		   void** buffer, int exact)
{
    int                 result;
    struct PipeStruct*  pipe;

    pipe = (struct PipeStruct*) malloc(sizeof(struct PipeStruct));
    if (pipe == NULL) return FALSE;

    pipe->func   = func;
    pipe->buffer = buffer;
    pipe->exact  = exact;
    pipe->error  = 0;

    result = FileTraverseFat(handle, fatcluster, SubdirTraverser,
			     (void**) &pipe);

    if (pipe->error) result = FALSE;

    free(pipe);

    return result;
}

int GetDirectory(RDWRHandle handle,
		 struct DirectoryPosition* position,
		 struct DirectoryEntry* direct)
{
    char buffer[BYTESPERSECTOR];

    if (ReadSectors(handle, 1, position->sector, buffer) == -1)
	return FALSE;

    memcpy(direct, &buffer[position->offset << 5],
	   sizeof(struct DirectoryEntry));

    return TRUE;
}

int DirectoryEquals(struct DirectoryPosition* pos1,
		    struct DirectoryPosition* pos2)
{
   return memcmp(pos1, pos2, sizeof(struct DirectoryPosition)) == 0;
}

int WriteDirectory(RDWRHandle handle, struct DirectoryPosition* pos,
		   struct DirectoryEntry* direct)
{
    char buffer[BYTESPERSECTOR];

    if (ReadSectors(handle, 1, pos->sector, buffer) == -1)
	return FALSE;

    memcpy(&buffer[pos->offset << 16], direct,
	   sizeof(struct DirectoryEntry));

    if (WriteSectors(handle, 1, pos->sector, buffer) == -1)
	return FALSE;

    return TRUE;
}
