/* Copyright (C) 2002 kjd */

/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <dir.h>
#include "ISFILE.H"   // int isfile (const char *filename);
#include "ISPRGAVL.H"


// local prototypes
char * tryPath(char *buffer, const char *drive, const char *path, const char *progname, const char *extlist);


/* This will search in the current directory, followed by
   each directory specified in the PATH environment variable
   for the program.  For each directory, for each extension
   specified in extlist (separated with a semicolon and no extra
   spaces), a check is made to see if the file exists (no
   check is made to ensure the file is actually executable).
   If the file exists (can be opened) then the complete path is returned,
   otherwise the search continues.  If all directories are
   searched and the program is not found then NULL is returned.
   Note: the string returned refers to a static buffer and
   is overwritten with each call.
   Note: this function may return NULL when in fact the program
   exists if there is not enough memory to copy the PATH into a
   scratch buffer.
*/
char * isProgramAvailable(const char *progname, const char *extlist)
{
  static char buffer[MAXPATH];
  char drive[MAXDRIVE], path[MAXPATH]; /* path is not MAXDIR on purpose */
  char *envPATH, *t;
  register char *p;

  // validate arguments, extlist can be NULL for no extension
  if ((progname == NULL) || !*progname) return NULL;

  // first try current drive and directory
  sprintf(drive, "%c:", 'A'+getdisk());
  *path = '\0';
  if (!getcurdir(0, path+1)) *path = '\\';  /* add root to path */

  if (tryPath(buffer, drive, path, progname, extlist) != NULL)
    return buffer;

  // copy PATH value into a non const location
  p = getenv("PATH");
  if (p == NULL)
    envPATH = NULL;
  else
  {
    if ((envPATH = (char *)malloc(strlen(p)+1)) != NULL)
      strcpy(envPATH, p);
    // else if we can't allocate memory so fail
  }

  t = envPATH;           // initialize to full PATH list
  while (t != NULL)
  {
    p = strchr(t, ';');  // find end of current path
    if (p != NULL)       // if more path sections follow
    {
      *p = '\0';         // then mark end of current one
      p++;               // and store start of next one
    }

    if (tryPath(buffer, NULL, t, progname, extlist) != NULL)
    {
      if (envPATH) free(envPATH);
      return buffer;
    }

    t = p;                // actually point to start of next one
  }
  if (envPATH) free(envPATH);

  return NULL;
}


static char * tryPath(char *buffer, const char *drive, const char *path, const char *progname, const char *extlist)
{
  char ext[MAXEXT];
  register const char *curExt;
  register char *p;

  if (buffer == NULL) return NULL;  // ensure not NULL

  // try each ; separated extension
  if (extlist == NULL)
    curExt = "";
  else
    curExt = extlist;               // initialize
  do
  {
    strncpy(ext, curExt, MAXEXT);   // copy over 1st extension (upto max ext size)
    ext[MAXEXT-1] = '\0';           // ensure string terminated
    p = strchr(ext, ';');           // see if we got more than 1 extension,
    if (p) *p = '\0';               // if so, truncate to end of 1st extension
    curExt = strchr(curExt, ';');   // prepare for next extension
    if (curExt) curExt++;           // point to start of it (if not end of list)

    // combine drive, path, progname, and extension
    fnmerge(buffer, drive, path, progname, ext);
    if (isfile(buffer)) return buffer;
  } while (curExt);

  return NULL;
}
